from javawrap import Color, Font, GridBagConstraints
from javawrap.Line2D import LineDouble
import MainHandle
import MajorObjects
import Normalization
import Plotting
import datetime
import java.awt.Dimension as Dimension
import java.awt.GridBagLayout as GridBagLayout
import java.awt.GridLayout as GridLayout
import javax.swing as swing
import javax.swing.BorderFactory as BorderFactory
import javax.swing.Box as Box
import javax.swing.JLabel as JLabel
import javax.swing.JPanel as JPanel
import javax.swing.JSlider as JSlider
import javax.swing.JTextField as JTextField
import javax.swing.SwingConstants as SwingConstants
import javax.swing.border.TitledBorder as TitledBorder
import org.jfree.data.xy.XYSeries as XYSeries

masterObject = MainHandle.masterObject

# standard borders for components:
emptyBorder = BorderFactory.createEmptyBorder(1, 1, 1, 1)
etchedBorder = BorderFactory.createEtchedBorder(swing.border.EtchedBorder.LOWERED)
loweredEtched = BorderFactory.createCompoundBorder(emptyBorder, etchedBorder)
RADIO_BOX_BG = Color.newColor(250, 250, 250)

# create the top-level Data Analysis tab
def createAnalysisPanel(debug=0):

	# create main frame:
	APanel = MajorObjects.NewPanel()
	APanel.setLayout(GridBagLayout())

	# row 1:
	# score distribution:
	(hpScoreDistributionPanel, compList) = makeHpScoreDistributionPanel()
	MainHandle.addToMasterObject(compList)

	AConstraint = MajorObjects.DefaultGridbagConstraints()
	AConstraint.gridx = 0
	AConstraint.gridy = 0
	AConstraint.weighty = 0.6
	AConstraint.fill = GridBagConstraints.BOTH
	AConstraint.anchor = GridBagConstraints.EAST
	APanel.add(hpScoreDistributionPanel, AConstraint)

	# row 2:
	AConstraint.fill = GridBagConstraints.HORIZONTAL
	AConstraint.gridx = 0
	AConstraint.weighty = 0.1
	AConstraint.gridy = 1
	(daSelectionPanel, compList) = makeSelectionPanel()
	APanel.add(daSelectionPanel, AConstraint)

	# row 3:
	(hitListPanel, compList) = makeHitListPanel()
	hitListPanel.setBackground(Color.GREEN)
	MainHandle.addToMasterObject(compList)
	AConstraint.fill = GridBagConstraints.BOTH
	AConstraint.weighty = 0.3
	AConstraint.gridy = 2
	APanel.add(hitListPanel, AConstraint)

	masterObject.setField('DataAnalysisPanel', APanel)
	APanel.repaint()
	return APanel

######### SCORING METHOD PANEL ####################
# This duplicates the functionality of the scoring method section of the Plate Scoring tab
def makeDaScoringMethodPanel():
	# scoring method:
	buttonList = ['b-score', 'z-score', 'robust z-score', 'PoC', 'raw score']
	scoreSelectBox = MajorObjects.RadioButtonUtils()
	scoreSelectBox.createRadioButtonGrouping(buttonList, 'Scoring method',
											MajorObjects.CustomActionListener(daScoreMethodActionListener),
											defaultSelect=1, nCol=2)

	scoreSelectBox.setButtonBgColor(Color.newColor(250, 250, 250))
	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.fill = GridBagConstraints.BOTH
	constraints.anchor = GridBagConstraints.WEST

	return (scoreSelectBox.panel, [['daScoringMethodBox', scoreSelectBox]])

# the ActionListeners below take a specified action when one of the buttons in this tab is clicked
def daScoreMethodActionListener(evt):
	oldScore = masterObject.getField('scoreMethod')
	evtStr = evt.getActionCommand()
	# return if no change occurred:
	if oldScore == evtStr:
		return

	#masterObject.dumpKeys()
	# invalidate the data to indicate that it needs to be re-normalized before it is displayed or written to a file
	Normalization.invalidatePlateAnalysis()

	# record the new button setting:
	masterObject.setField('scoreMethod', evtStr)
	scoreSelectBox = masterObject.getField('scoreSelectBox')
	scoreSelectBox.clickButton(evtStr)

	updateDataAnalysisPanel()

def daHpSelectionGroupingActionListener(evt):
	masterObject.setField('hpSelectionGrouping', evt.getActionCommand())
	masterObject.setField('Compute hairpin scores', 1)

	# click the button on the Plate Scoring tab:
	evtStr = evt.getActionCommand()
	hpsBox = masterObject.getField('hpSelGroupingBox')
	hpsBox.clickButton(evtStr)

	updateDataAnalysisPanel()

def daHpBatchGroupingActionListener(evt):
	masterObject.setField('hpBatchGrouping', evt.getActionCommand())
	masterObject.setField('Compute hairpin scores', 1)

	# click the button on the Plate Scoring tab:
	evtStr = evt.getActionCommand()
	bgBox = masterObject.getField('hpBatchGroupingBox')
	bgBox.clickButton(evtStr)

	updateDataAnalysisPanel()

def daSubsetActionListener(evt):
	oldSubset = masterObject.getField('plateSubset')
	evtStr = evt.getActionCommand()
	# return if no change occurred:
	if oldSubset == evtStr:
		return

	#masterObject.setField('plateSubset', evtStr)

	# click the button on the Plate Scoring tab:
	subBox = masterObject.getField('subsetBox')
	subBox.clickButton(evtStr)

	updateDataAnalysisPanel()

def updateDataAnalysisPanel():
	# update the panel:
	Normalization.processAnalysisPlateset()    # process the plate set data
	Normalization.computeHairpinScores()       # compute hairpin scores for the plate set
	hpData = masterObject.getField('hpData')   # update the hairpin data in the master object

	# put all hairpin scores into one big list:
	scoreList = []
	maxStd = 0.0
	for hpKey in hpData.keys():
		hpId = hpKey.split(':')[0]
		cloneId = hpKey.split(':')[1]
		for screen in hpData[hpKey].keys():
			for batch in hpData[hpKey][screen].keys():
				for pCond in hpData[hpKey][screen][batch].keys():
					for selStat in hpData[hpKey][screen][batch][pCond].keys():
						hpPtr = hpData[hpKey][screen][batch][pCond][selStat]  # pointer to the hairpin
						hpInfo = hpPtr['info'].split('\t')
						scoreList.append([hpPtr['nMean'], hpPtr['nStd'], hpInfo[0], hpId, pCond, selStat,
										  batch, cloneId, hpInfo[9], hpInfo[1]])
						if hpPtr['nStd'] > maxStd:
							maxStd = hpPtr['nStd']

	# sort the list based on the mean normalized value for each hairpin (for the score distribution panel curve)
	scoreList.sort(element1)   

	masterObject.setField('hairpinScoreList', scoreList)
	confThreshSlider = masterObject.getField('confThreshSlider')
	confThreshSlider.updateLimits(0.0, maxStd, maxStd / 2.0)
	confThresh = confThreshSlider.getValue()
	# update the hairpin score distribution:
	updateHpScoreDistribution(confThresh)

	APanel = masterObject.getField('DataAnalysisPanel')
	APanel.repaint()

######### SCORE DISTRIBUTION PANEL ################
def makeHpScoreDistributionPanel():
	
	hpScoreDistributionPanel = JPanel(GridBagLayout())

	# low score threshold slider panel (to the left of the score distribution)
	(ltSlidePanel, compList) = makeLowThreshSlidePanel()
	#ltSlidePanel.setBackground(Color.newColor(100,200,255))
	MainHandle.addToMasterObject(compList)

	# high score threshold slider panel (to the left of the score distribution)
	(htSlidePanel, compList) = makeHighThreshSlidePanel()
	#htSlidePanel.setBackground(Color.newColor(100,200,255))
	MainHandle.addToMasterObject(compList)

	# the panel for the hairpin score distribution
	(dpPanel, compList) = makeDistPlotPanel()
	#dpPanel.setBackground(Color.newColor(100,200,255))
	MainHandle.addToMasterObject(compList)

	# buttons for saving high- and low-end hits:
	saveLoHpHitsButton = MajorObjects.BorderedButton('Save low-end hits', saveLoEndHits, layout='gridBag')
	saveLoHpHitsButton.setBackground(Color.newColor(255, 100, 100))
	saveHiHpHitsButton = MajorObjects.BorderedButton('Save high-end hits', saveHiEndHits, layout='gridBag')
	saveHiHpHitsButton.setBackground(Color.newColor(100, 255, 100))

	# Box for low slider, dot plot, and high slider:
	topBox = hpScoreDistributionPanel
	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.gridy = 0
	constraints.fill = GridBagConstraints.NONE
	constraints.anchor = GridBagConstraints.CENTER
	constraints.gridheight = 1
	constraints.gridx = 0
	topBox.add(Box.createHorizontalGlue(), constraints)   # column 1 spacer

	constraints.fill = GridBagConstraints.HORIZONTAL
	constraints.gridx = GridBagConstraints.RELATIVE
	constraints.gridheight = 1
	topBox.add(ltSlidePanel, constraints)                 # lowThresh slider

	#constraints.fill = GridBagConstraints.VERTICAL
	constraints.fill = GridBagConstraints.NONE
	constraints.gridx = GridBagConstraints.RELATIVE
	constraints.gridheight = 1
	topBox.add(Box.createHorizontalGlue(), constraints)   # column 3 spacer

	constraints.fill = GridBagConstraints.BOTH
	constraints.gridx = GridBagConstraints.RELATIVE
	yStart = 8
	constraints.gridheight = yStart
	constraints.gridwidth = 3   # 2
	topBox.add(dpPanel, constraints)                      # distribution panel

	#constraints.fill = GridBagConstraints.VERTICAL
	constraints.fill = GridBagConstraints.NONE
	constraints.gridwidth = 1
	constraints.gridx = GridBagConstraints.RELATIVE
	constraints.gridheight = 1
	topBox.add(Box.createHorizontalGlue(), constraints)   # column 7 spacer

	constraints.fill = GridBagConstraints.HORIZONTAL
	constraints.gridx = GridBagConstraints.RELATIVE
	constraints.gridheight = 2
	topBox.add(htSlidePanel, constraints)                 # highThresh slider

	#constraints.fill = GridBagConstraints.VERTICAL
	constraints.fill = GridBagConstraints.BOTH
	constraints.gridx = GridBagConstraints.RELATIVE
	constraints.gridheight = 1
	topBox.add(Box.createHorizontalGlue(), constraints)   # column 9 spacer

	################################
	# spacer:
	constraints.gridx = 1
	constraints.gridy = 5
	constraints.gridheight = 1
	constraints.gridwidth = 1
	constraints.fill = GridBagConstraints.VERTICAL
	constraints.anchor = GridBagConstraints.NORTH
 	topBox.add(Box.createHorizontalGlue(), constraints)

	# add buttons below threshold sliders:
	constraints.gridheight = 1
	constraints.gridx = 0
	constraints.gridy = yStart - 2
	constraints.gridwidth = 2
	constraints.fill = GridBagConstraints.BOTH
	constraints.anchor = GridBagConstraints.WEST
	topBox.add(saveLoHpHitsButton, constraints)

	constraints.gridx = 7
	constraints.gridheight = 1
	constraints.gridwidth = 2
	constraints.fill = GridBagConstraints.BOTH
	constraints.anchor = GridBagConstraints.EAST
	topBox.add(saveHiHpHitsButton, constraints)

	# spacer:
	constraints.weighty = 0.1
	constraints.gridx = 0
	constraints.gridy = yStart
	constraints.gridheight = 1
	constraints.gridwidth = 9
	constraints.fill = GridBagConstraints.HORIZONTAL
	constraints.anchor = GridBagConstraints.CENTER
	box1 = Box.createHorizontalBox()
	box1.setBorder(BorderFactory.createLineBorder(Color.GRAY))
	topBox.add(box1, constraints)

	return (hpScoreDistributionPanel, [['hpScoreDistributionPanel', hpScoreDistributionPanel]])

def makeSelectionPanel():
	# make the Score StdDev threshold slider panel
	(ctSlidePanel, compList) = makeConfThreshSlidePanel()
	#ctSlidePanel.setBackground(Color.newColor(100,200,255))
	MainHandle.addToMasterObject(compList)

	daSelectionPanel = JPanel(GridBagLayout())
	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.gridheight = 1
	constraints.gridwidth = 1
	constraints.fill = GridBagConstraints.NONE
	constraints.anchor = GridBagConstraints.WEST

	# add selection/batch grouping buttons:
	(flagPanel, compList) = makeFlagPanel()
	constraints.gridx = 0
	constraints.gridwidth = 4
	constraints.fill = GridBagConstraints.BOTH
	daSelectionPanel.add(flagPanel, constraints)   # add the button groups in the middle laft

	constraints.gridx = 6
	constraints.gridheight = 1
	constraints.gridwidth = 1
	constraints.fill = GridBagConstraints.NONE
	constraints.anchor = GridBagConstraints.CENTER
	daSelectionPanel.add(ctSlidePanel, constraints)    # add the conidence slider toward the middle

	geneButtonBox = makeGeneButtonBox()
	constraints.gridx = 7
	constraints.fill = GridBagConstraints.BOTH
	daSelectionPanel.add(geneButtonBox, constraints)   # add the "Save Gene-level results" button 

	return (daSelectionPanel, [['geneButtonBox', geneButtonBox]])

def makeGeneButtonBox():
	# the "save Gene-level results" button holder
	geneButtonPanel = JPanel(GridBagLayout())
	geneButtonPanel.setBorder(TitledBorder('Gene-level results'))
	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.gridheight = 1
	constraints.gridwidth = 1
	constraints.fill = GridBagConstraints.HORIZONTAL
	constraints.anchor = GridBagConstraints.CENTER

	# make the button
	defGeneSaveButton = MajorObjects.BorderedButton('Save Gene-level results', saveGeneLevel, layout='gridBag')

	constraints.gridx = 0
	constraints.gridy = 0
	geneButtonPanel.add(defGeneSaveButton, constraints)
	constraints.gridy = 1

	# Removed non-functional 'Export for WGA' button
	#exportToWgaButton = MajorObjects.BorderedButton('Export for WGA', exportWGA, layout='gridBag')
	#geneButtonPanel.add(exportToWgaButton, constraints)

	return geneButtonPanel

def saveGeneLevel(evt):
	# action taken whe the "Save Gene-level results" button is pressed
	print 'Saving gene-level results'
	# reprocess the analysis plateset, if necessary:
	if masterObject.getField('Update analysis'):
		Normalization.processAnalysisPlateset()

	# choose the output file name, if fout returns as None, return without saving:
	fout = MajorObjects.chooseFile()
	if fout == None:
		return

	# make column headers:
	fout.write('# NOTE: plate-level info is at the bottom of this file\n\n')
	fout.write('# GENE-level analysis results\n')
	fout.write('# File generated by:\t%s\n' % masterObject.getField('uName'))
	fout.write('# Created on:\t%s\n' % datetime.datetime.now().strftime('%d-%b-%Y %H:%M:%S'))
	fout.write('# Software version:\t%s\n\n' % masterObject.getField('Software version'))

	# build strings to write in the output file:
	# Low threshold slider value:
	ltThreshSlider = masterObject.getField('ltThreshSlider')
	try:
		iThreshMin = float(ltThreshSlider.getThresh())
		#print 'iThreshMin: ', ltThreshSlider.getThresh()
		iThrMinStr = '%5.2f' % iThreshMin
	except:
		iThreshMin = None
		iThrMinStr = '<not specified>'

	# High threshold slider value:
	htThreshSlider = masterObject.getField('htThreshSlider')
	try:
		iThreshMax = float(htThreshSlider.getThresh())
		#print 'iThreshMax: ', htThreshSlider.getThresh()
		iThrMaxStr = '%5.2f' % iThreshMax
	except:
		iThreshMax = None
		iThrMaxStr = '<not specified>'

	# Confidence/StdDev threshold slider value:
	try:
		confThreshSlider = masterObject.getField('confThreshSlider')
		confThresh = confThreshSlider.getDisplayedValue()
		#print 'confThresh: ', confThreshSlider.getDisplayedValue()
		confThrStr = '%6.2f' % confThresh
	except:
		confThresh = None
		confThrStr = '<not specified>'

	# compute hairpin scores, if necessary:
	if masterObject.getField('Compute hairpin scores'):
		Normalization.computeHairpinScores()
	# compute gene scores:
	geneData = gatherGeneLevel(iThreshMin, iThreshMax, confThresh)

	# write info to he output file:
	fout.write('# Plate subset selection:\t%s\n' % masterObject.getField('plateSubset'))
	fout.write('# Scoring method:\t%s\n' % masterObject.getField('scoreMethod'))
	fout.write('# Scoring granularity:\t%s\n' % masterObject.getField('quad'))
	fout.write('# Use of control wells:\t%s\n' % masterObject.getField('controlWells'))
	fout.write('# Hairpin selection grouping:\t%s\n' % masterObject.getField('hpSelectionGrouping'))
	fout.write('# Gene-level batch grouping:\t%s\n' % masterObject.getField('hpBatchGrouping'))
	fout.write('#  Gene score High threshold:\t%s\n' % iThrMaxStr)
	fout.write('#  Gene score Low threshold:\t%s\n' % iThrMinStr)
	fout.write('#  Gene confidence threshold:\t%s\n\n' % confThrStr)

	fout.write('screen\tbatch\tcondition\tselection\tgeneID\tsymbol\tdescription\ttaxon\tlow score\tlow conf.\thigh score\thigh conf.\t# hp\t# confident hp\n')
	# write the actual gene-level results into the file:
	for geneId in geneData.keys():
		for screen in geneData[geneId].keys():
			for batch in geneData[geneId][screen].keys():
				for pCond in geneData[geneId][screen][batch].keys():
					for selStat in geneData[geneId][screen][batch][pCond].keys():
						gPtr = geneData[geneId][screen][batch][pCond][selStat]
						gScore = gPtr['score']
						str1 = '%s\t%s\t%s\t%s' % (screen, batch, pCond, selStat)
						str2 = '%s\t%s\t%s\t%s' % (geneId, gScore.symbol, gScore.desc, gScore.taxon)
						str3 = '%6.3f\t%d\t%6.3f\t%d\t%d\t%d' % (gScore.minScore, gScore.minConf,
																 gScore.maxScore, gScore.maxConf,
																 gScore.nHp, gScore.nConf)
						fout.write('%s\t%s\t%s\n' % (str1, str2, str3))

	fout.close()   # close the output file

def gatherGeneLevel(tMin, tMax, tConf):
	# gather up the gene-level data into the geneData dictionary to allow fast output of the gene-level results
	cloneMap = masterObject.getField('cloneMap')
	hpData = masterObject.getField('hpData')
	hpList = hpData.keys()     # get the list of all hairpins used
	hpList.sort()              # sort them

	geneData = {}      # initialize the geneData dictionary
	for hpKey in hpList:
		#hpName = hpKey.split(':')[0]
		cloneId = hpKey.split(':')[1]     # get the cloneID for the hairpin
		cloneInfo = cloneMap[cloneId]     # get the clone info
		geneId = cloneInfo.geneId         # get the gene ID
		geneData.setdefault(geneId, {})   # initialize storage for the gene
		#print 'hpName,cloneId, cloneInfo,geneId: ',hpName,cloneId, cloneInfo,geneId
		# gather all of the hairpins objects in the heirarchy: geneID,screen,batch, condition, selection status, clone ID
		for screen in hpData[hpKey].keys():
			geneData[geneId].setdefault(screen, {})
			for batch in hpData[hpKey][screen].keys():
				geneData[geneId][screen].setdefault(batch, {})
				for pCond in hpData[hpKey][screen][batch].keys():
					geneData[geneId][screen][batch].setdefault(pCond, {})
					for selStat in hpData[hpKey][screen][batch][pCond].keys():
						geneData[geneId][screen][batch][pCond].setdefault(selStat, {})
						hpPtr = hpData[hpKey][screen][batch][pCond][selStat]
						geneData[geneId][screen][batch][pCond][selStat][cloneId] = hpPtr


	# gather the gene data for each gene
	for geneId in geneData.keys():
		for screen in geneData[geneId].keys():
			for batch in geneData[geneId][screen].keys():
				for pCond in geneData[geneId][screen][batch].keys():
					for selStat in geneData[geneId][screen][batch][pCond].keys():
						gPtr = geneData[geneId][screen][batch][pCond][selStat]    # pointer to the gene object
						gScore = GeneScore()                                      # GeneScore object for score computation
						gScore.computeGeneScore(tMin, tMax, tConf, gPtr)          # compute the gene score
						gPtr['score'] = gScore                                    # save the gene score in the gene pointer object

	return geneData

# GeneScore class description: an object tha gets created only for computing the gene-level scoring
class GeneScore:
	def __init__(self):
		self.symbol = None        # the usual gene-level information
		self.desc = None          #        "
		self.taxon = None         #        "
		self.gContig = None       #        "
		self.gStart = None        #        "
		self.gEnd = None          #        "
		self.gStrand = None       #        "
		self.gType = None         #        "
		self.transSeq = None      #        "
		self.nHp = None           # number of hairpins in the screen that target this gene
		self.nConf = None         # "confident" hairpin count
		self.minHp = None         # 
		self.maxHp = None         # 
		self.minScore = None      # minimum score for all hairpins
		self.maxScore = None      # maximum score for all hairpins
		self.minConf = None       # number of hairpins that pass both the minimum score and confidence thresholds
		self.maxConf = None       # number of hairpins that pass both the maximum score and confidence thresholds
		self.loHits = []          # low-end hairpin "hits"
		self.hiHits = []          # high-end hairpin "hits"

	def computeGeneScore(self, tMin, tMax, tConf, hpSet):
		# NOTE: the scoring algorithm is as follows:
		# minScore = minimum score for all hairpins
		# minConf = number of hairpins that pass both the minimum score and confidence thresholds
		# minFConf = minConf/nHp (minConf as a fraction of the number of hairpins)
		# maxScore = maximum score for all hairpins
		# maxConf = number of hairpins that pass both the maximum score and confidence thresholds
		# maxFConf = maxConf/nHp (maxConf as a fraction of the number of hairpins)

		for hp in hpSet.keys():
			hpScore = hpSet[hp]['nMean']
			hpConf = hpSet[hp]['nStd']
			if self.nHp == None:
				# first hairpin...
				# fill in gene data from 'info' field of the first hairpin:
				info = hpSet[hp]['info'].split('\t')
				self.symbol = info[0]
				self.desc = info[1]
				self.taxon = info[3]
				self.gContig = info[5]
				self.gStart = info[6]
				self.gEnd = info[7]
				self.gStrand = info[8]
				self.gType = info[9]

				# initialize score fields:
				self.nHp = 1
				self.minHp = hp
				self.maxHp = hp
				self.minScore = hpScore
				self.maxScore = hpScore
				if hpConf <= tConf:
					self.nConf = 1
					if hpScore <= tMin:
						self.loHits.append([hp, '%6.4f' % hpScore, '%6.4f' % hpConf])
						self.minConf = 1
						self.maxConf = 0
					elif hpScore >= tMax:
						self.hiHits.append([hp, '%6.4f' % hpScore, '%6.4f' % hpConf])
						self.minConf = 0
						self.maxConf = 1
					else:
						self.minConf = 0
						self.maxConf = 0
				else:
					self.nConf = 0
					self.minConf = 0
					self.maxConf = 0
			else:
				# increment the number of hairpins
				self.nHp += 1
				# update min/max scores:
				if hpScore < self.minScore:
					self.minScore = hpScore
					self.minHp = hp
				if hpScore > self.maxScore:
					self.maxScore = hpScore
					self.maxHp = hp

				# update min/max/total confidence counts
				if hpScore <= tMin and hpConf <= tConf:
					self.loHits.append([hp, '%6.4f' % hpScore, '%6.4f' % hpConf])
					self.minConf += 1
					self.nConf += 1
				elif hpScore >= tMax and hpConf <= tConf:
					self.hiHits.append([hp, '%6.4f' % hpScore, '%6.4f' % hpConf])
					self.maxConf += 1
					self.nConf += 1
				elif hpConf <= tConf:
					self.nConf += 1

		# finalize minFConf and maxFConf:
		self.minFConf = float(self.minConf) / self.nHp
		self.maxFConf = float(self.minConf) / self.nHp


def exportWGA(evt):
	# placeholder for output for Whole Genome Analysis (WGA) input file
	print 'Exporting for WGA analysis'

def listSizes(evt):
	# for debugging 
	print 'Button pressed'
	obj = masterObject.getField('hpScoreDistributionPanel')
	print 'hpScoreDistributionPanel:'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('hpDistChart')
	print 'hpDistChart:'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('ltSlidePanel')
	print 'ltSlidePanel'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('htSlidePanel')
	print 'htSlidePanel'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('daScoringMethodBox')
	print 'daScoringMethodBox'
	print '       size:', obj.panel.getSize()
	print '  pref size:', obj.panel.getPreferredSize()

	obj = masterObject.getField('ctSlidePanel')
	print 'ctSlidePanel'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('hitListPanel')
	print 'hitListPanel'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('KD_hitsPanel')
	print 'KD_hitsPanel'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

	obj = masterObject.getField('KU_hitsPanel')
	print 'KU_hitsPanel'
	print '       size:', obj.getSize()
	print '  pref size:', obj.getPreferredSize()

def saveLoEndHits(evt):
	# action that occurs when the "Save low-end hits" button is pressed
	# choose the output file name, if fout returns as None, return without saving:
	fout = MajorObjects.chooseFile()
	if fout == None:
		return

	taggedPassSet = masterObject.getField('taggedPassSet') # get the tagged set of hairpins that pass the threshold tests
	tKeys = taggedPassSet.keys()
	tKeys.sort()
	hpDistChart = masterObject.getField('hpDistChart')     # hairpin score distribution chart
	passSeries = hpDistChart.getPointsetSeries('pass')     # get the list of points that pass the threshold test

	ltThreshSlider = masterObject.getField('ltThreshSlider')   # get the low thresh slider
	lThresh = ltThreshSlider.getThresh()                       # get the low threshold
	xVal = findXVal(passSeries, lThresh)                       # since the list is sorted
	confThreshSlider = masterObject.getField('confThreshSlider')   # get the confidence threshold
	confThresh = confThreshSlider.getValue() / 100.0               # get the value of the threshold from the slider

	fout.write('# Low-end (knockdown) hairpin hits\n')
	fout.write('# File generated by:\t%s\n' % masterObject.getField('uName'))
	fout.write('# Created on:\t%s\n' % datetime.datetime.now().strftime('%d-%b-%Y %H:%M:%S'))      # "now"
	fout.write('# Software version:\t%s\n\n' % masterObject.getField('Software version'))
	fout.write('# Plate subset selection:\t%s\n' % masterObject.getField('plateSubset'))
	fout.write('# Scoring method:\t%s\n' % masterObject.getField('scoreMethod'))
	fout.write('# Scoring granularity:\t%s\n' % masterObject.getField('quad'))
	fout.write('# Use of control wells:\t%s\n' % masterObject.getField('controlWells'))
	fout.write('# Hairpin selection grouping:\t%s\n' % masterObject.getField('hpSelectionGrouping'))
	fout.write('# Low score threshold:\t%6.2f\n' % lThresh)                # low score threshold
	fout.write('# Score StdDev threshold:\t%6.2f\n\n' % confThresh)        # low-end confidence threshold

	# write the file header:
	fout.write('Symbol\thairpin\tbatch\tselection\tcondition\tscore\tscore StdDev\tvirusID\tpref name\tvType\n')

	nHits = 0
	for tag in tKeys:
		if tag <= xVal:
			x = taggedPassSet[tag]   # get the info for this "tag"
			fout.write('%s\t%s\t%s\t%s\t%s\t%6.3f\t%6.3f\t%s\t%s\t%s\n' % (x[2], x[3], x[6], x[5], x[4],
																		   float(x[0]), float(x[1]),
																		   x[7], x[9], x[8]))
			nHits += 1
		else:
			break

	#print 'low-end hits: %d'%nHits
	fout.close()
	return

def saveHiEndHits(evt):
	# choose the output file name, if fout returns as None, return without saving:
	fout = MajorObjects.chooseFile()
	if fout == None:
		return

	taggedPassSet = masterObject.getField('taggedPassSet')
	tKeys = taggedPassSet.keys()
	tKeys.sort()
	hpDistChart = masterObject.getField('hpDistChart')
	passSeries = hpDistChart.getPointsetSeries('pass')

	htThreshSlider = masterObject.getField('htThreshSlider')
	hThresh = htThreshSlider.getThresh()
	xVal = findXVal(passSeries, hThresh)
	confThreshSlider = masterObject.getField('confThreshSlider')
	confThresh = confThreshSlider.getValue() / 100.0

	fout.write('# High-end (increased proliferation) hairpin hits\n')
	fout.write('# File generated by:\t%s\n' % masterObject.getField('uName'))
	fout.write('# Created on:\t%s\n' % datetime.datetime.now().strftime('%d-%b-%Y %H:%M:%S'))
	fout.write('# Software version:\t%s\n\n' % masterObject.getField('Software version'))
	fout.write('# Plate subset selection:\t%s\n' % masterObject.getField('plateSubset'))
	fout.write('# Scoring method:\t%s\n' % masterObject.getField('scoreMethod'))
	fout.write('# Scoring granularity:\t%s\n' % masterObject.getField('quad'))
	fout.write('# Use of control wells:\t%s\n' % masterObject.getField('controlWells'))
	fout.write('# Hairpin selection grouping:\t%s\n' % masterObject.getField('hpSelectionGrouping'))
	fout.write('# High score threshold:\t%6.2f\n' % hThresh)
	fout.write('# Score StdDev threshold:\t%6.2f\n\n' % confThresh)

	# header:
	fout.write('Symbol\thairpin\tbatch\tselection\tcondition\tscore\tscore StdDev\tvirusID\tpref name\tvType\n')

	nHits = 0
	for tag in tKeys:
		if tag >= xVal:
			x = taggedPassSet[tag]
			fout.write('%s\t%s\t%s\t%s\t%s\t%6.3f\t%6.3f\t%s\t%s\t%s\n' % (x[2], x[3], x[6], x[5], x[4],
																		   float(x[0]), float(x[1]),
																		   x[7], x[9], x[8]))
			nHits += 1

	#print 'high-end hits: %d'%nHits
	fout.close()
	return

def makeFlagPanel():
	flagPanel = JPanel(GridBagLayout())
	#flagPanel.setBackground(Color.GRAY)
	(daSelPanel, compList) = makeDaSubsetBox()
	MainHandle.addToMasterObject(compList)
	(daSelGroupPanel, compList) = makeDaHpSelectionGroupingBox()
	MainHandle.addToMasterObject(compList)
	(daBatchGroupPanel, compList) = makeDaHpBatchGroupingBox()
	MainHandle.addToMasterObject(compList)
	(daScoringMethodPanel, compList) = makeDaScoringMethodPanel()
	MainHandle.addToMasterObject(compList)

	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.weightx = 0.33
	constraints.gridx = 0
	constraints.gridy = 0
	constraints.gridheight = 2
	constraints.fill = GridBagConstraints.HORIZONTAL
	constraints.anchor = GridBagConstraints.WEST

	constraints.gridwidth = 1
	constraints.gridheight = 2
	flagPanel.add(daSelPanel, constraints)

	constraints.anchor = GridBagConstraints.EAST
	constraints.gridwidth = 1
	constraints.gridheight = 1
	constraints.gridx = 2
	flagPanel.add(daSelGroupPanel, constraints)

	constraints.gridy = 1
	constraints.gridx = 2
	flagPanel.add(daBatchGroupPanel, constraints)

	constraints.gridheight = 2
	constraints.gridy = 0
	constraints.gridx = 3
	flagPanel.add(daScoringMethodPanel, constraints)

	return (flagPanel, [['flagPanel', flagPanel]])

def makeDaHpSelectionGroupingBox():
	font = Font.Ariel12Bold
	ifont = Font.Ariel12BoldItalic

	buttonList = ['combine puro+/puro-', 'separate puro+/puro-']
	hpSelGroupingBox = MajorObjects.RadioButtonUtils()
	hpSelGroupingBox.createRadioButtonGrouping(buttonList, 'Selection (puro) grouping',
										MajorObjects.CustomActionListener(daHpSelectionGroupingActionListener),
										defaultSelect=1)
	hpSelGroupingBox.setButtonBgColor(RADIO_BOX_BG)

	try:
		masterObject.getField('hpSelectionGrouping')
	except:
		masterObject.setField('hpSelectionGrouping', 'separate puro+/puro-')

	return (hpSelGroupingBox.panel, [['daHpSelGroupingBox', hpSelGroupingBox],
							  ['daHpSelGroupingPanel', hpSelGroupingBox.panel]])

def makeDaHpBatchGroupingBox():
	font = Font.Ariel12Bold
	ifont = Font.Ariel12BoldItalic

	buttonList = ['combine batches', 'keep separate']
	hpBatchGroupingBox = MajorObjects.RadioButtonUtils()
	hpBatchGroupingBox.createRadioButtonGrouping(buttonList, 'Batch grouping',
										MajorObjects.CustomActionListener(daHpBatchGroupingActionListener),
										defaultSelect=1)
	hpBatchGroupingBox.setButtonBgColor(RADIO_BOX_BG)

	try:
		masterObject.getField('hpBatchGrouping')
	except:
		masterObject.setField('hpBatchGrouping', 'keep separate')

	return (hpBatchGroupingBox.panel, [['daHpBatchGroupingBox', hpBatchGroupingBox],
							  ['daHpBatchGroupingPanel', hpBatchGroupingBox.panel]])

def makeDaSubsetBox():
	font = Font.Ariel12Bold
	ifont = Font.Ariel12BoldItalic

	buttonList = ['only selection +', 'only selection -', 'all plates']
	subsetBox = MajorObjects.RadioButtonUtils()
	subsetBox.createRadioButtonGrouping(buttonList, 'Plate subset selection',
							  MajorObjects.CustomActionListener(daSubsetActionListener),
							  defaultSelect=2)
	subsetBox.setButtonBgColor(RADIO_BOX_BG)

	masterObject.setField('plateSubset', None)

	return (subsetBox.panel, [['daSubsetBox', subsetBox],
							  ['daSubsetPanel', subsetBox.panel]])


### score distribution subpanels ############
def makeLowThreshSlidePanel():
	ltSlidePanel = JPanel(GridBagLayout())
	ltSlidePanel.setPreferredSize(Dimension(150, 300))
	#ltSlidePanel.setBackground(Color.newColor(255,128,128))
	eBorder = BorderFactory.createLineBorder(Color.GRAY)
	#tBorder = BorderFactory.createTitledBorder(eBorder, 'Low score threshold:',
	#										   TitledBorder.CENTER, TitledBorder.TOP)
	ltSlidePanel.setBorder(eBorder)
	lowThreshSlider = MajorObjects.ThresholdSliderWithLimits('Low score threshold:', 100, 100, 0.0, 100.0, 0.0, side=JTextField.LEFT)
	lowThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(lowThreshTextActionListener))
	lowThreshSlider.addSliderSideEffect(updateLowThreshold)

	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.anchor = GridBagConstraints.NORTH
	constraints.fill = GridBagConstraints.NONE
	ltSlidePanel.add(lowThreshSlider, constraints)

	# add hit-counter panel:
	lowHitCounter = JPanel()
	lowText = JLabel('low-end hits:')
	lowText.setHorizontalAlignment(SwingConstants.RIGHT)
	lowHitCounter.add(lowText)
	lowHitCount = JTextField(5)
	lowHitCount.setHorizontalAlignment(SwingConstants.LEFT)
	lowHitCounter.add(lowHitCount)
	constraints.gridy = 1
	ltSlidePanel.add(lowHitCounter, constraints)

	return(ltSlidePanel, [['ltSlidePanel', ltSlidePanel],
						  ['ltThreshSlider', lowThreshSlider],
						  ['lowHitCount', lowHitCount]])

def makeHighThreshSlidePanel():
	htSlidePanel = JPanel(GridBagLayout())
	htSlidePanel.setPreferredSize(Dimension(150, 300))
	#htSlidePanel.setBackground(Color.newColor(255,128,128))
	eBorder = BorderFactory.createLineBorder(Color.GRAY)
	#tBorder = BorderFactory.createTitledBorder(eBorder, 'High score threshold:',
	#										   TitledBorder.CENTER, TitledBorder.TOP)
	htSlidePanel.setBorder(eBorder)
	highThreshSlider = MajorObjects.ThresholdSliderWithLimits('High score threshold:', 100, 100, 0.0, 100.0, 0.0)
	highThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(highThreshTextActionListener))
	highThreshSlider.addSliderSideEffect(updateHighThreshold)

	constraints = MajorObjects.DefaultGridbagConstraints()
	constraints.anchor = GridBagConstraints.NORTH
	constraints.fill = GridBagConstraints.NONE
	htSlidePanel.add(highThreshSlider, constraints)

	# add hit-counter panel:
	highHitCounter = JPanel()
	highText = JLabel('high-end hits:')
	highText.setHorizontalAlignment(SwingConstants.RIGHT)
	highHitCounter.add(highText)
	highHitCount = JTextField(5)
	highHitCount.setHorizontalAlignment(SwingConstants.LEFT)
	highHitCounter.add(highHitCount)
	constraints.gridy = 1
	htSlidePanel.add(highHitCounter, constraints)

	return(htSlidePanel, [['htSlidePanel', htSlidePanel],
						  ['htThreshSlider', highThreshSlider],
						  ['highHitCount', highHitCount]])

def makeConfThreshSlidePanel():
	ctSlidePanel = JPanel(GridBagLayout())
	ctSlidePanel.setPreferredSize(Dimension(300, 65))
	eBorder = BorderFactory.createEmptyBorder()
	lBorder = BorderFactory.createLineBorder(Color.GRAY)
	cBorder = BorderFactory.createCompoundBorder(lBorder, eBorder)
	ctSlidePanel.setBorder(cBorder)
	confThreshSlider = MajorObjects.ThresholdSliderWithLimits('Score StdDev threshold:', 100, 100, 0.0, 1.0, 0.5, orient=JSlider.HORIZONTAL)
	confThreshSlider.addSliderSideEffect(updateHpScoreDistribution)
	confThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(confThreshTextActionListener))

	ctConstraints = MajorObjects.DefaultGridbagConstraints()
	ctSlidePanel.add(confThreshSlider, ctConstraints)
	return(ctSlidePanel, [['ctSlidePanel', ctSlidePanel],
					  ['confThreshSlider', confThreshSlider]])

def makeConfThreshSlidePanel_orig():
	ctSlidePanel = JPanel()
	ctSlidePanel.setPreferredSize(Dimension(300, 65))
	#ctSlidePanel.setBackground(Color.newColor(255,128,128))
	eBorder = BorderFactory.createEmptyBorder()
	lBorder = BorderFactory.createLineBorder(Color.GRAY)
	cBorder = BorderFactory.createCompoundBorder(lBorder, eBorder)
	#tBorder = BorderFactory.createTitledBorder(eBorder, 'Score StdDev threshold:',
	#										   TitledBorder.CENTER, TitledBorder.TOP)
	ctSlidePanel.setBorder(cBorder)
	confThreshSlider = MajorObjects.ThresholdSlider('Score StdDev threshold:', 100, 100, 0.0, 1.0, 0.5, orient=JSlider.HORIZONTAL, headerGap=8)
	confThreshSlider.addSliderSideEffect(updateHpScoreDistribution)
	confThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(confThreshTextActionListener))

	ctSlidePanel.add(confThreshSlider)
	return(ctSlidePanel, [['ctSlidePanel', ctSlidePanel],
					  ['confThreshSlider', confThreshSlider]])

def makeDistPlotPanel():
	#dpPanel = JPanel()
	#dpPanel.setBackground(Color.newColor(255,128,255))

	hpDistChart = Plotting.NewScatterChart('Hairpin score distribution', None, 'score', 700, 400)
	hpDistChart.xyplot.getDomainAxis().setTickLabelsVisible(0)
	hpDistChart.xyplot.getDomainAxis().setTickMarksVisible(0)
	#dpPanel.add(hpDistChart)

	return (hpDistChart, [['hpDistChart', hpDistChart]])
	#return (dpPanel, [['dpPanel',dpPanel],
	#				  ['hpDistChart', hpDistChart]])

#################################################### End Score Distrbution Panel

######### HIT LIST PANEL ################
def makeHitListPanel():
	hitListPanel = JPanel(GridLayout(1, 2))
	(KD_hitsPanel, compList) = makeKDhitsPanel()
	MainHandle.addToMasterObject(compList)
	(KU_hitsPanel, compList) = makeKUhitsPanel()
	MainHandle.addToMasterObject(compList)

	hitListPanel.add(KD_hitsPanel)
	hitListPanel.add(KU_hitsPanel)

	return (hitListPanel, [['hitListPanel', hitListPanel]])

#################################
def makeKDhitsPanel():
	colNames = ['Symbol', 'Screen:Batch', 'Sel', 'Condition', 'Score']
	kdListPane = MajorObjects.NormTablePane(0)
	kdListPane.setColumnNames(colNames)
	kdListPane.setPreferredColumnWidths([150, 120, 30, 150, 100])
	kdListPane.setPreferredSize(Dimension(458, 230))
	mBorder = BorderFactory.createMatteBorder(2, 2, 2, 2, Color.newColor(255, 100, 100))
	kdListPane.setBorder(BorderFactory.createCompoundBorder(mBorder, etchedBorder))
	kdListPane.setColumnHorizontalAlignment(JLabel.CENTER)

	return (kdListPane, [['KD_hitsPanel', kdListPane]])

def makeKUhitsPanel():
	colNames = ['Symbol', 'Screen:Batch', 'Sel', 'Condition', 'Score']
	kuListPane = MajorObjects.NormTablePane(0)
	kuListPane.setColumnNames(colNames)
	kuListPane.setPreferredColumnWidths([150, 120, 30, 150, 100])
	kuListPane.setPreferredSize(Dimension(458, 230))
	mBorder = BorderFactory.createMatteBorder(2, 2, 2, 2, Color.newColor(100, 255, 100))
	kuListPane.setBorder(BorderFactory.createCompoundBorder(mBorder, etchedBorder))
	kuListPane.setColumnHorizontalAlignment(JLabel.CENTER)

	return (kuListPane, [['KU_hitsPanel', kuListPane]])

#################################################### End Hit List Panel
def getDataAnalysisData(dummy):
	scoring = masterObject.getField('scoreMethod')
	print '    Scoring method: %s' % scoring

	if scoring == 'Infection efficiency':
		# set the scoring method to the default:
		normScoreSelectBox = masterObject.getField('scoreSelectBox')
		normScoreSelectBox.clickDefaultButton()
		daScoreSelectBox = masterObject.getField('daScoringMethodBox')
		daScoreSelectBox.clickDefaultButton()

	DataAnalysisPanel = masterObject.getField('DataAnalysisPanel')
	hpDistChart = masterObject.getField('hpDistChart')
	hpDistChart.clearChart()
	#vPlateListPane = masterObject.getField('daVpListPanel')
	#vPlateListPane.clearList()

	DataAnalysisPanel.refresh()

	#repCompChart = masterObject.getField('plateReplicateChart')
	#print 'repCompChart size:', repCompChart.getSize()
	hpDistChart	 = masterObject.getField('hpDistChart')
	#print ' hpDistChart size:', hpDistChart.getSize()

	# reprocess the analysis plateset, if necessary:
	if masterObject.getField('Update analysis'):
		Normalization.processAnalysisPlateset()

	# compute the hairpin scores:
	masterObject.setField('resetHpThresh', 1)
	if masterObject.getField('Compute hairpin scores'):
		Normalization.computeHairpinScores()

	# update the virus plateset list:
	#updateDAVirusPlateList()

	hpData = masterObject.getField('hpData')
	hpList = hpData.keys()

	# put all hairpin scores into one big list:
	scoreList = []
	maxStd = 0.0
	for hpKey in hpData.keys():
		hpId = hpKey.split(':')[0]
		cloneId = hpKey.split(':')[1]
		for screen in hpData[hpKey].keys():
			for batch in hpData[hpKey][screen].keys():
				for pCond in hpData[hpKey][screen][batch].keys():
					for selStat in hpData[hpKey][screen][batch][pCond].keys():
						hpPtr = hpData[hpKey][screen][batch][pCond][selStat]
						hpInfo = hpPtr['info'].split('\t')
						scoreList.append([hpPtr['nMean'], hpPtr['nStd'], hpInfo[0], hpId, pCond, selStat,
										  batch, cloneId, hpInfo[9], hpInfo[1]])
						if hpPtr['nStd'] > maxStd:
							maxStd = hpPtr['nStd']

	#for hp in hpList:
	#	hpScores = hpData[hp].getHpScores()
	#	for cond in hpScores.keys():
	#		for selStat in hpScores[cond].keys():
	#			for batch in hpScores[cond][selStat].keys():
	#				scoreTemp = hpScores[cond][selStat][batch]
	#				score = [scoreTemp[0], scoreTemp[1], hpData[hp].symbol, hp, cond, selStat, batch,
	#						 hpData[hp].virusId, hpData[hp].vType, hpData[hp].prefName]
	#				if score[0]!=None:
	#					scoreList.append(score)      # [score, scoreStd]
	#					if score[1]>maxStd:
	#						maxStd=score[1]

	print 'Total score count: %d' % len(scoreList)
	#print 'Max StdDev: ', maxStd

	scoreList.sort(element1)

	masterObject.setField('hairpinScoreList', scoreList)
	confThreshSlider = masterObject.getField('confThreshSlider')
	confThreshSlider.updateLimits(0.0, maxStd, maxStd / 2.0)
	confThresh = confThreshSlider.getValue()
	# update the hairpin score distribution:
	updateHpScoreDistribution(confThresh)

def updateHpScoreDistribution(confThresh):
	ctSlidePanel = masterObject.getField('ctSlidePanel')
	#print ctSlidePanel.getSize()
	scoreList = masterObject.getField('hairpinScoreList')

	# build 2 XYSeries, one with scores <= the confidence threshold (passing)
	#  and one for failing scores. The x-value of the elements are monotonically increasing
	#  and unique across the two sets:
	taggedPassSet = {}
	passSeries = XYSeries('pass')
	failSeries = XYSeries('fail')
	allSeries = XYSeries('all')

	# horizontal and vertical components of the low and high threshold crosshairs:
	ltVert = XYSeries('ltVert')
	ltHztl = XYSeries('ltHztl')
	htVert = XYSeries('htVert')
	htHztl = XYSeries('htHztl')
	#stdSeries = XYSeries('std')
	for idx in range(len(scoreList)):
		if scoreList[idx][1] <= confThresh:
			passSeries.add(idx, scoreList[idx][0])
			taggedPassSet[idx] = scoreList[idx]
		else:
			failSeries.add(idx, scoreList[idx][0])
		allSeries.add(idx, scoreList[idx][0])
		#stdSeries.add(idx, scoreList[idx][1])

	# initialize display with the sorted list of scores:
	hpDistChart = masterObject.getField('hpDistChart')
	hpDistChart.clearChart()
	hpDistChart.addPointset('fail', failSeries, color=Color.RED)
	hpDistChart.setPointsetAlpha('fail', 0.5)
	#hpDistChart.setPointsetSize('fail', 1)
	hpDistChart.setPointsetShape('fail', LineDouble(0, -2.0, 0, 2.0))
	hpDistChart.addPointset('pass', passSeries, color=Color.BLACK)
	hpDistChart.setSeriesShapeVisibility('pass', 0)
	hpDistChart.setSeriesLineVisibility('pass', 1)
	#hpDistChart.addPointset('std', stdSeries, color=Color.GREEN)

	# save the full dataset of scores:
	masterObject.setField('fullHpScoreList', allSeries)

	# save tagged list of hairpin scores passing the confidence threshold:
	masterObject.setField('taggedPassSet', taggedPassSet)

	# reset the thresholds if the hairpin scores have been changed:
	if masterObject.getField('resetHpThresh'):

		# add empty crosshair series:
		hpDistChart.addPointset('ltVert', ltVert, color=Color.newColor(200, 0, 0))
		hpDistChart.setSeriesShapeVisibility('ltVert', 0)
		hpDistChart.setSeriesLineVisibility('ltVert', 1)
		hpDistChart.addPointset('ltHztl', ltHztl, color=Color.BLACK)
		hpDistChart.setSeriesShapeVisibility('ltHztl', 0)
		hpDistChart.setSeriesLineVisibility('ltHztl', 1)
		hpDistChart.addPointset('htVert', htVert, color=Color.newColor(0, 200, 0))
		hpDistChart.setSeriesShapeVisibility('htVert', 0)
		hpDistChart.setSeriesLineVisibility('htVert', 1)
		hpDistChart.addPointset('htHztl', htHztl, color=Color.BLACK)
		hpDistChart.setSeriesShapeVisibility('htHztl', 0)
		hpDistChart.setSeriesLineVisibility('htHztl', 1)

		# initialize the score threshold sliders to the 5th and 95th percentile of all scores:
		#nScores = allSeries.getItemCount()
		#idx1 = nScores/100.0
		#minScore = allSeries.getDataItem(0).getY()
		#maxScore = allSeries.getDataItem(nScores-1).getY()
		#score5 = allSeries.getDataItem(int(idx1*5.0))
		#score50 = allSeries.getDataItem(int(idx1*50.0))
		#score95 = allSeries.getDataItem(int(idx1*95.0))
		nScores = passSeries.getItemCount()
		idx1 = nScores / 100.0
		minScore = passSeries.getDataItem(0).getY()
		maxScore = passSeries.getDataItem(nScores - 1).getY()
		score5 = passSeries.getDataItem(int(idx1 * 5.0))
		score50 = passSeries.getDataItem(int(idx1 * 50.0))
		score95 = passSeries.getDataItem(int(idx1 * 95.0))

		ltThreshSlider = masterObject.getField('ltThreshSlider')
		ltThreshSlider.updateLimits(minScore, score50.getY(), score5.getY())
		#print 'Updating ltThresh limits: %6.2f, %6.2f, %6.2f' % (minScore, score50.getY(), score5.getY())
		htThreshSlider = masterObject.getField('htThreshSlider')
		htThreshSlider.updateLimits(score50.getY(), maxScore, score95.getY())
		#print 'Updating htThresh limits: %6.2f, %6.2f, %6.2f' % (score50.getY(), maxScore, score95.getY())

		# hpScoreRange contains: [minX, maxX, minScore, maxScore]
		minX = 0
		maxX = allSeries.getItemCount()
		masterObject.setField('hpScoreRange', [minX, maxX, minScore, maxScore])
		#print 'Low thresh:',ltThreshSlider.getValue()
		updateLowThreshold(ltThreshSlider.getThresh())
		#print 'High thresh:',htThreshSlider.getValue()
		updateHighThreshold(htThreshSlider.getThresh())

# sorts a list of lists by the first element in each list:
def element1(a, b):
	if a[0] < b[0]:
		return - 1
	elif a[0] > b[0]:
		return 1
	else:
		return 0

def updateLowThreshold(lThresh):
	# update low threhold crosshairs:
	taggedPassSet = masterObject.getField('taggedPassSet')
	tKeys = taggedPassSet.keys()
	tKeys.sort()
	hpDistChart = masterObject.getField('hpDistChart')
	ltHztl = hpDistChart.getPointsetSeries('ltHztl')
	ltVert = hpDistChart.getPointsetSeries('ltVert')
	try:
		[minX, maxX, minScore, maxScore] = masterObject.getField('hpScoreRange')
	except:
		return

	passSeries = hpDistChart.getPointsetSeries('pass') # get all of the data that passes the confidence threshold test
	xVal = findXVal(passSeries, lThresh)               # find the index of the highest value less than the threshold
	kdListPane = masterObject.getField('KD_hitsPanel') # get the panel that displays the list of hairpin hits
	kdListPane.clearList()                             # clear the list to start
	nHits = 0
	for tag in tKeys:
		# fill in the rows of the low-end hit list
		if tag <= xVal:
			x = taggedPassSet[tag]
			if x[5].endswith('YES') or x[5].endswith('PLUS'):
				selStat = '+'
			elif x[5].endswith('+/-'):
				selStat = '+/-'
			else:
				selStat = '-'

			# symbol, batch, score
			if x[2] == None:
				row = [x[3], x[6], selStat, x[4], '%6.3f' % float(x[0])]
			else:
				row = [x[2], x[6], selStat, x[4], '%6.3f' % float(x[0])]
			kdListPane.addRow(row, tag)  # add another row to the hit list
			nHits += 1
		else:
			break

	lowHitCount = masterObject.getField('lowHitCount') # get the low-end hit count text field
	lowHitCount.setText(' %d ' % nHits)                # display the number of low-end hits

	x20th = (maxX - minX) / 20.0          # 1/20 of the x-range (for crosshair)
	x0h = max(xVal - x20th, minX)
	x1h = min(xVal + x20th, maxX)

	# draw the low threhold 'crosshair' in 4 parts:
	ltHztl.clear()             # black horizontal line:
	ltHztl.add(x0h, lThresh)   # left half
	ltHztl.add(x1h, lThresh)   # right half
	ltVert.clear()             # read verical line
	ltVert.add(xVal, minScore) # lower part of line
	ltVert.add(xVal, maxScore) # upper part of line
	
def updateHighThreshold(hThresh):
	# update high threshold crosshairs:
	taggedPassSet = masterObject.getField('taggedPassSet')
	tKeys = taggedPassSet.keys()
	tKeys.sort()
	tKeys.reverse()
	hpDistChart = masterObject.getField('hpDistChart')
	htHztl = hpDistChart.getPointsetSeries('htHztl')
	htVert = hpDistChart.getPointsetSeries('htVert')
	try:
		[minX, maxX, minScore, maxScore] = masterObject.getField('hpScoreRange')
	except:
		return

	passSeries = hpDistChart.getPointsetSeries('pass') # get all of the data that passes the confidence threshold test
	xVal = findXVal(passSeries, hThresh)               # find the index of the highest value less than the threshold
	kuListPane = masterObject.getField('KU_hitsPanel') # get the panel that displays the list of hairpin hits
	kuListPane.clearList()                             # clear the list to start
	nHits = 0
	for tag in tKeys:
		# fill in the rows of the low-end hit list
		if tag >= xVal:
			x = taggedPassSet[tag]
			if x[5].endswith('YES'):
				selStat = '+'
			else:
				selStat = '-'

			# symbol, batch, score
			if x[2] == None:
				row = [x[3], x[6], selStat, x[4], '%6.3f' % float(x[0])]
			else:
				row = [x[2], x[6], selStat, x[4], '%6.3f' % float(x[0])]
			kuListPane.addRow(row, tag)  # add another row to the hit list
			nHits += 1
		else:
			break

	highHitCount = masterObject.getField('highHitCount')   # get the low-end hit count text field
	highHitCount.setText(' %d ' % nHits)

	x20th = (maxX - minX) / 20.0       # 1/20 of the x-range (for crosshair)
	x0h = max(xVal - x20th, minX)
	x1h = min(xVal + x20th, maxX)

	# draw the low threhold 'crosshair' in 4 parts:
	htHztl.clear()             # black horizontal line:
	htHztl.add(x0h, hThresh)   # left half
	htHztl.add(x1h, hThresh)   # right half
	htVert.clear()             # read verical line
	htVert.add(xVal, minScore) # lower part of line
	htVert.add(xVal, maxScore) # upper part of line

def lowThreshTextActionListener(evt):
	# this Action Listener allows the threshold to be typed into the low threshold text box
	newVal = float(evt.getActionCommand())
	sliderObj = masterObject.getField('ltThreshSlider')
	slider = sliderObj.slider
	minThresh = sliderObj.minThresh
	step = sliderObj.stepSize

	# compute the position of the slider:
	sVal = int((newVal - minThresh) / step)
	# when the slider is set, the slider action listener fires, and the graphics are updated
	slider.setValue(sVal)      

def highThreshTextActionListener(evt):
	# this Action Listener allows the threshold to be typed into the high threshold text box
	newVal = float(evt.getActionCommand())
	sliderObj = masterObject.getField('htThreshSlider')
	slider = sliderObj.slider
	minThresh = sliderObj.minThresh
	step = sliderObj.stepSize

	# compute the position of the slider:
	sVal = int((newVal - minThresh) / step)
	# when the slider is set, the slider action listener fires, and the graphics are updated
	slider.setValue(sVal)

def confThreshTextActionListener(evt):
	# this Action Listener allows the threshold to be typed into the confidence threshold text box	
	newVal = float(evt.getActionCommand())
	sliderObj = masterObject.getField('confThreshSlider')
	slider = sliderObj.slider
	minThresh = sliderObj.minThresh
	step = sliderObj.stepSize

	# compute the position of the slider:
	sVal = int((newVal - minThresh) / step)
	# when the slider is set, the slider action listener fires, and the graphics are updated
	slider.setValue(sVal)


def findXVal(series, yValue, direction='low2high', idxStart=None):
	# search an XYSeries for the xValue associated with a
	# yValue in the series. If the search direction is 'low2high' (default),
	# the series is searched from low to high, otherwise it is searched
	# from high to low. If a value is given for xStart, the search will start
	# from that index. The returned xValue is the for which the next yValue is
	# greater than (or less than, if the search is not low-to-high) the input
	# yValue. If no valid result is found, the highest xValue (or lowest for
	# high-to-low) is returned.
	nPoints = series.getItemCount()
	if direction == 'low2high':
		# default result:
		lastPt = series.getDataItem(nPoints - 1)
		# low-to-high search:
		if idxStart == None:
			idx = 0
		else:
			idx = idxStart

		while idx < nPoints:
			pt = series.getDataItem(idx)
			if pt.getY() > yValue:
				break
			else:
				lastPt = pt
				idx += 1

	else:
		# default result:
		lastPt = series.getDataItem(0)
		# high-to-low search:
		if idxStart == None:
			idx = nPoints - 1
		else:
			idx = idxStart

		while idx >= 0:
			pt = series.getDataItem(idx)
			if pt.getY() < yValue:
				break
			else:
				lastPt = pt
				idx -= 1

	return lastPt.getX()

# Not used:
def updateDAVirusPlateList():
	# get the list of virus plates:
	vPlateSet = masterObject.getField('VirusPlates')  ## <<<<<<< get from vPlateSet map
	# get the analysis plateSet:
	analysisSet = masterObject.getField('analysisPlateset')

	vPlateListPane = masterObject.getField('daVpListPanel')
	vPlateListPane.clearList()

	aPlateList = analysisSet.getPlateIdList()
	vPlateRows = {}
	for plateID in aPlateList:
		plate = analysisSet.getPlate(plateID)
		if not plate.use:
			continue
		plateId = plate.plateId
		#vPlateId = plate.vPlateId
		#screenId = plate.screenId
		#batchId = plate.batchId
		vPlateName = plate.vPlateName
		#screenName = plate.screenName
		batchName = plate.batchName
		selStat = plate.selStat
		cond = plate.cond

		vPlateRows.setdefault(vPlateName, {})
		vPlateRows[vPlateName].setdefault(batchName, {})
		vPlateRows[vPlateName][batchName].setdefault(selStat, {})
		vPlateRows[vPlateName][batchName][selStat].setdefault(cond, [])
		vPlateRows[vPlateName][batchName][selStat][cond].append(plateId)


	vPlateNameList = vPlateRows.keys()
	vPlateNameList.sort()
	for vPlateName in vPlateNameList:
		for batchName in vPlateRows[vPlateName].keys():
			for selStat in vPlateRows[vPlateName][batchName].keys():
				for cond in vPlateRows[vPlateName][batchName][selStat].keys():
					plateList = vPlateRows[vPlateName][batchName][selStat][cond]
					row = [vPlateName, batchName, selStat, cond, len(plateList)]
					vPlateListPane.addRow(row, plateList)
