from javawrap import Color, GridBagConstraints, Font
import MainHandle
import MajorObjects
import Normalization
import Plotting
import java.awt.Dimension as Dimension
import java.awt.GridBagLayout as GridBagLayout
import java.awt.Insets as Insets
import javax.swing as swing
import javax.swing.BorderFactory as BorderFactory
import javax.swing.Box as Box
import javax.swing.BoxLayout as BoxLayout
import javax.swing.JLabel as JLabel
import javax.swing.JPanel as JPanel
import javax.swing.JSlider as JSlider
import javax.swing.JTextField as JTextField
import javax.swing.ListSelectionModel as ListSelectionModel
import org.jfree.data.xy.XYSeries as XYSeries
import org.jfree.data.xy.XYSeriesCollection as XYSeriesCollection

masterObject = MainHandle.masterObject

# colors for InfectionEfficiency dotplot:
HP_COLOR = Color.BLUE
EMP_COLOR = Color.GRAY
CTL_COLOR = Color.RED
PGW_COLOR = Color.newColor(0, 180, 130)
#PGW_COLOR = Color.WHITE

DEF_BG_COLOR = Color.newColor(0.9, 0.2, 0.2)

# create the BIN_EDGES for the histogram:
nBins = 41
maxBin = 2.0
BIN_EDGES = []
GT_STRING = '>2'
binStep = maxBin / (nBins - 1)
for i in range(nBins):
	BIN_EDGES.append(binStep * i)

# standard borders for components:
emptyBorder = BorderFactory.createEmptyBorder(1, 1, 1, 1)
etchedBorder = BorderFactory.createEtchedBorder(swing.border.EtchedBorder.LOWERED)
loweredEtched = BorderFactory.createCompoundBorder(emptyBorder, etchedBorder)

# Create the main Infection Efficiency tab
def createInfectionEfficiencyPanel(debug=0):

	# create main frame:
	IePanel = MajorObjects.NewPanel()
	masterObject.setField('IePanel', IePanel)
	IePanel.setLayout(GridBagLayout())

	# dot-plot box:
	(graphBox, compList) = makeGraphBox()
	MainHandle.addToMasterObject(compList)

	# IE histogram:
	(histoPanel, compList) = makeIEHistogram()
	MainHandle.addToMasterObject(compList)

	# IE Histogram type selection box:
	(ieHistTypeBox, compList) = makeIeHistTypeBox()
	MainHandle.addToMasterObject(compList)

	# slideBox contains the two sliders:
	(slideBox, compList) = makeIEThresholdSliderBox()
	MainHandle.addToMasterObject(compList)

	# create text box displaying count of removed wells by category:
	(statsBox, compList) = makeWellCountStatsBox()
	MainHandle.addToMasterObject(compList)

	# make a listBox of the virus plates available, with reset button
	(vpListPanel, compList) = makeVpListPanel(debug)
	MainHandle.addToMasterObject(compList)

	(lowCutoffBox, compList) = makeLowCutoffSliderBox()
	MainHandle.addToMasterObject(compList)

	# basic, single-item GridBagConstraints:
	unitConstraints = MajorObjects.DefaultGridbagConstraints()

	# JPanel 'holders' for components:
	gbPanel = JPanel(GridBagLayout())
	unitConstraints.insets = Insets(5, 5, 5, 5)
	gbPanel.add(graphBox, unitConstraints)
	masterObject.setField('gbPanel', gbPanel)

	# virus plate info panel
	vpPanel = JPanel(GridBagLayout())
	unitConstraints.insets = Insets(5, 0, 5, 0)
	# add the virus plate list panel
	vpPanel.add(vpListPanel, unitConstraints)
	masterObject.setField('vpPanel', vpPanel)

	unitConstraints.resetDefaults()
	
	# Bottom portion of the window (IE histogram, sliders, etc.)
	iePanel = JPanel(GridBagLayout())
	unitConstraints.fill = GridBagConstraints.HORIZONTAL
	unitConstraints.insets = Insets(0, 10, 0, 0)
	iePanel.add(ieHistTypeBox, unitConstraints)
	masterObject.setField('iePanel', iePanel)

	unitConstraints.resetDefaults()

	# histogram panel 
	hpPanel = JPanel(GridBagLayout())
	hpPanel.add(histoPanel, unitConstraints)
	masterObject.setField('hpPanel', hpPanel)

	unitConstraints.insets = Insets(5, 5, 5, 5)

	# slider panel
	slPanel = JPanel(GridBagLayout())
	slPanel.setBorder(loweredEtched)
	slPanel.add(slideBox, unitConstraints)
	masterObject.setField('slPanel', slPanel)

	# statistics panel
	stPanel = JPanel(GridBagLayout())
	unitConstraints.fill = GridBagConstraints.HORIZONTAL
	stPanel.add(statsBox, unitConstraints)
	masterObject.setField('stPanel', stPanel)

	# GridBagConstraints object:
	IeConstraint = GridBagConstraints.newGridBagConstraints()
	# default constraints:
	IeConstraint.gridheight = 1

	# component size parameters:
	#dotBoxX = 600
	#dotBoxY = 600

	# TOP ROW::::::::::::::
	# graphBox (top left)
	IeConstraint.anchor = GridBagConstraints.WEST
	IeConstraint.fill = GridBagConstraints.BOTH
	IeConstraint.gridwidth = 2
	IeConstraint.gridx = 0
	IeConstraint.gridy = 0
	IeConstraint.weightx = 0.66
	IeConstraint.weighty = 0.70
	IePanel.add(gbPanel, IeConstraint)

	# vpListPanel (top right)
	IeConstraint.anchor = GridBagConstraints.EAST
	IeConstraint.fill = GridBagConstraints.BOTH
	IeConstraint.gridwidth = 2
	IeConstraint.gridx = GridBagConstraints.RELATIVE
	IeConstraint.gridy = 0
	IeConstraint.weightx = GridBagConstraints.REMAINDER
	IePanel.add(vpPanel, IeConstraint)

	# BOTTOM ROW::::::::::::::
	IeConstraint.gridheight = 2

	# Add components from left to right
	# ieHistTypeBox and container
	IeConstraint.gridwidth = 1
	IeConstraint.fill = GridBagConstraints.BOTH
	IeConstraint.anchor = GridBagConstraints.SOUTH
	IeConstraint.weighty = GridBagConstraints.REMAINDER
	IeConstraint.weightx = GridBagConstraints.REMAINDER
	IeConstraint.gridx = 0
	IeConstraint.gridy = 1
	IePanel.add(iePanel, IeConstraint)

	# histoPanel (histogram)
	IeConstraint.gridwidth = 1
	IeConstraint.anchor = GridBagConstraints.SOUTH
	IeConstraint.fill = GridBagConstraints.BOTH
	IeConstraint.gridx = GridBagConstraints.RELATIVE
	IeConstraint.gridy = 1
	IeConstraint.weightx = 0.6
	IePanel.add(hpPanel, IeConstraint)

	# slideBox (min and max treshold sliders)
	IeConstraint.gridwidth = 1
	IeConstraint.weightx = GridBagConstraints.REMAINDER
	IeConstraint.anchor = GridBagConstraints.SOUTH
	IeConstraint.fill = GridBagConstraints.VERTICAL
	IeConstraint.gridx = GridBagConstraints.RELATIVE
	IeConstraint.gridy = 1
	IePanel.add(slPanel, IeConstraint)

	# statsBox (statistics box)
	IeConstraint.gridwidth = 1
	IeConstraint.gridheight = 1
	IeConstraint.weightx = GridBagConstraints.REMAINDER
	IeConstraint.anchor = GridBagConstraints.NORTH
	IeConstraint.fill = GridBagConstraints.BOTH
	IeConstraint.gridx = GridBagConstraints.RELATIVE
	IeConstraint.gridy = 1
	IePanel.add(stPanel, IeConstraint)

	# low cutoff threshold slider (added below the statistics box)
	IeConstraint.gridwidth = 1
	IeConstraint.weightx = 0.2
	IeConstraint.anchor = GridBagConstraints.WEST
	IeConstraint.fill = GridBagConstraints.BOTH
	IeConstraint.gridy = 2
	IePanel.add(lowCutoffBox, IeConstraint)

	# repaint the panel
	IePanel.repaint()
	return IePanel

# construction of the box with well count statistics:
def makeWellCountStatsBox():
	font = Font.Ariel12Bold
	ifont = Font.Ariel12BoldItalic

	# layout parameters for the box:
	unitConstraints = GridBagConstraints.newGridBagConstraints()
	unitConstraints.gridx = 0
	unitConstraints.gridy = 0
	unitConstraints.gridwidth = GridBagConstraints.REMAINDER
	unitConstraints.gridheight = GridBagConstraints.REMAINDER
	unitConstraints.weightx = 1.0
	unitConstraints.weighty = 1.0
	unitConstraints.anchor = GridBagConstraints.WEST
	unitConstraints.fill = GridBagConstraints.BOTH
	unitConstraints.insets = Insets(0, 0, 0, 0)
	unitConstraints.ipadx = 0
	unitConstraints.ipady = 0

	# make the box
	statsBox = JPanel(GridBagLayout())
	statsBox.setBorder(loweredEtched)

	# left side layout parameters
	cLeft = GridBagConstraints.newGridBagConstraints()
	cLeft.fill = GridBagConstraints.HORIZONTAL
	cLeft.weightx = 0.25
	cLeft.gridwidth = 1
	cLeft.gridx = 0

	# right side layout parameters
	cRight = GridBagConstraints.newGridBagConstraints()
	cRight.fill = GridBagConstraints.HORIZONTAL
	cRight.weightx = 0.75
	cRight.gridwidth = GridBagConstraints.REMAINDER

	# title:
	titleBox = JLabel('Wells outside threshold limits:')
	titleBox.setHorizontalAlignment(JTextField.LEFT)
	titleBox.setFont(ifont)
	cRight.ipadx = 5
	cRight.ipady = 7
	cRight.gridx = 0
	cRight.gridy = 0
	statsBox.add(titleBox, cRight)

	# valid hairpin well stats
	hpText = JLabel('valid hp: ')  # label 
	hpText.setHorizontalAlignment(JTextField.RIGHT)
	hpText.setFont(ifont)
	cLeft.ipadx = 3
	cLeft.ipady = 7
	cLeft.gridx = 0
	cLeft.gridy = 1
	statsBox.add(hpText, cLeft)

	hpStat = JTextField('')        # data
	hpStat.setHorizontalAlignment(JTextField.CENTER)
	hpStat.setFont(font)
	cRight.ipadx = 0
	cRight.gridx = 1
	cRight.gridy = 1
	statsBox.add(hpStat, cRight)

	# control hairpin well stats
	ctText = JLabel('control: ')   # label
	ctText.setHorizontalAlignment(JTextField.RIGHT)
	ctText.setFont(ifont)
	cLeft.gridx = 0
	cLeft.gridy = 2
	statsBox.add(ctText, cLeft)

	ctStat = JTextField('')        # data
	ctStat.setHorizontalAlignment(JTextField.CENTER)
	ctStat.setFont(font)
	cRight.gridx = 1
	cRight.gridy = 2
	statsBox.add(ctStat, cRight)

	# EMPTY well stats:
	emText = JLabel('empty: ')     # label
	emText.setHorizontalAlignment(JTextField.RIGHT)
	emText.setFont(ifont)
	cLeft.gridx = 0
	cLeft.gridy = 3
	statsBox.add(emText, cLeft)

	emStat = JTextField('')        # data
	emStat.setHorizontalAlignment(JTextField.CENTER)
	emStat.setFont(font)
	cRight.gridx = 1
	cRight.gridy = 3
	statsBox.add(emStat, cRight)

	# pgw well stats:
	pgwText = JLabel('pgw: ')      # label
	pgwText.setHorizontalAlignment(JTextField.RIGHT)
	pgwText.setFont(ifont)
	cLeft.gridx = 0
	cLeft.gridy = 4
	statsBox.add(pgwText, cLeft)

	pgwStat = JTextField('')       # data
	pgwStat.setHorizontalAlignment(JTextField.CENTER)
	pgwStat.setFont(font)
	cRight.gridx = 1
	cRight.gridy = 4
	statsBox.add(pgwStat, cRight)
	statsBox.getLayout().setConstraints(statsBox, unitConstraints)

	# return the box object and pointers to all of the text boxes 
	return (statsBox, [['statsBox', statsBox], ['hpStat', hpStat],
					   ['ctStat', ctStat], ['emStat', emStat], ['pgwStat', pgwStat]])

# IE scatter plot object
def makeGraphBox():
	graphBox = Plotting.NewScatterChart('Infection Efficiency', 'Puro -', 'Puro +', 533, 600)

	return (graphBox, [['graphBox', graphBox]])

# IE histogram object
def makeIEHistogram():
	histoPanel = Plotting.NewHistoChart('Infection Efficiency Distribution', 'Infection Efficiency',
							   '', 533, 200)
	histoPanel.setBorder(loweredEtched)

	return (histoPanel, [['histoPanel', histoPanel]])

# virus plate list panel
def makeVpListPanel(debug):
	# virus plate listBox container:
	vpListPanel = MajorObjects.NewPanel()
	vpListPanel.setBorder(loweredEtched)
	vpListPanel.setLayout(BoxLayout(vpListPanel, BoxLayout.Y_AXIS))

	# Column names and preferred widths:
	colNames = ['Virus Plate', 'screen', 'batch', 'cond', '#puro+', '#puro-', 'hp%', 'empty%']
	widthList = [115, 45, 45, 45, 50, 50, 55, 55]

	# display the table in a scrollable pane:
	vpListBox = MajorObjects.NormTablePane()
	vpListBox.setColumnNames(colNames)
	vpListBox.setPreferredColumnWidths(widthList)
	vpListBox.table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION)
	vpListBox.addListSelectionListener(MajorObjects.TableListSelectionListener(vpListBox.table,
																  vpListActionListener))
	vpListBox.addHeaderListener(MajorObjects.HeaderMouseListener(vpListBox.table, vpColumnSelect))
	vpListBox.setBorder(loweredEtched)
	vpListBox.setPreferredSize(Dimension(450, 580))

	vpListPanel.addComponent('vpListBox', vpListBox)

	# ExamineAllPlates button:
	localButtonPanel = JPanel()
	resetIEListButton = MajorObjects.BorderedButton('Examine all plates', resetPlateList)
	resetIEListButton.button.setEnabled(0)  # initialize to disabled
	localButtonPanel.add(resetIEListButton)

	# Axis autoscale on/off button
	ieAxisScaleButton = MajorObjects.BorderedButton('Axis autoscale off', toggleAxisAutoscale)
	ieAxisScaleButton.button.setEnabled(1)  # initialize to enabled
	ieAxisScaleButton.button.setBackground(Color.newColor(100, 255, 100))
	localButtonPanel.add(ieAxisScaleButton)

	# button is externally accessible as resetIEListButton
	vpListPanel.addComponent('resetIEListButton', localButtonPanel)

	# return the panel object as well as pointers to externally accessible objects
	return (vpListPanel, [['vpListPanel', vpListPanel], ['vpListBox', vpListBox],
						  ['vpListBoxTable', vpListBox.table],
						  ['resetIEListButton', resetIEListButton.button],
						  ['ieAxisScaleButton', ieAxisScaleButton.button],
						  ['toggleAxisAutoscale', 'off'],
						  ['vpColumnSortIdx', -1]])

# Infection efficiency slider box object
def makeIEThresholdSliderBox():
	slideBox = Box(BoxLayout.X_AXIS)
	slideBox.setBackground(Color.newColor(0.9, 0.2, 0.2))

	# create the IE min slider
	minThreshSlider = MajorObjects.ThresholdSlider('IE min:', 45, 100, 0.0, 1.0, 0.25)
	minThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(minThreshTextActionListener))
	slideBox.add(minThreshSlider)

	# spacer between sliders:
	slideBox.add(Box.createRigidArea(Dimension(10, 50)))

	# create the IE max slider
	maxThreshSlider = MajorObjects.ThresholdSlider('IE max:', 45, 100, 1.0, 4.0, 4.0)
	maxThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(maxThreshTextActionListener))
	slideBox.add(maxThreshSlider)

	slideBox.setAlignmentX(0.5)
	slideBox.setMaximumSize(Dimension(100, 150))

	# return the slider box object and pointers to externally accessible objects
	return(slideBox, [['slideBox', slideBox],
					  ['minThreshSlider', minThreshSlider], ['IE min:', 0.25],
					  ['maxThreshSlider', maxThreshSlider], ['IE max:', 4.0]])

# Infection efficiency absolute minimum threshold slider
def makeLowCutoffSliderBox():
	coSlidePanel = Box(BoxLayout.X_AXIS)
	coSlidePanel.setPreferredSize(Dimension(300, 65))
	lowCutoffThreshSlider = MajorObjects.ThresholdSlider('Low end cutoff:', 100, 100, 0.0, 1.0, 0.0,
											orient=JSlider.HORIZONTAL, headerGap=3, textWidth=10, sliderSteps=1000)
	lowCutoffThreshSlider.threshTextField.addActionListener(MajorObjects.CustomActionListener(lowCutoffThreshTextActionListener))
	coSlidePanel.add(lowCutoffThreshSlider)

	# return the slider panel object and pointers to externally accessible objects
	return(coSlidePanel, [['coSlidePanel', coSlidePanel],
					  ['lowCutoffThreshSlider', lowCutoffThreshSlider]])

# Infection efficiency histogram "type" box
def makeIeHistTypeBox():
	buttonList = ['Total', 'Type', 'Non-empty']
	ieHistTypeBox = MajorObjects.RadioButtonUtils()
	ieHistTypeBox.createRadioButtonGrouping(buttonList, 'Percent of...',
										MajorObjects.CustomActionListener(ieHistTypeActionListener),
										defaultSelect=0)

	masterObject.setField('ieHistType', 'Total')

	# return the box object and pointers to externally accessible objects
	return (ieHistTypeBox.panel, [['ieHistTypeBox', ieHistTypeBox],
							  ['ieHistTypeBoxPanel', ieHistTypeBox.panel]])

# action listener fired when text is typed in the minimum threshold slider text box
def minThreshTextActionListener(evt):
	Normalization.invalidatePlateAnalysis()
	newVal = float(evt.getActionCommand())
	source = evt.getSource()
	sliderObj = masterObject.getField('minThreshSlider')
	slider = sliderObj.slider
	minThresh = sliderObj.minThresh
	maxThresh = sliderObj.maxThresh
	step = sliderObj.stepSize

	# compute the position of the slider:
	sVal = int((newVal - minThresh) / step)
	slider.setValue(sVal)

# action listener fired when text is typed in the absolute minimum threshold slider text box
def lowCutoffThreshTextActionListener(evt):
	Normalization.invalidatePlateAnalysis()
	newVal = float(evt.getActionCommand())
	maxData = masterObject.getField('maxPuroMinus')
	source = evt.getSource()
	sliderObj = masterObject.getField('lowCutoffThreshSlider')
	slider = sliderObj.slider

	# compute/update the position of the slider:
	sliderObj.updateSliderValue(newVal / maxData)

# action listener fired when text is typed in the maximum threshold slider text box
def maxThreshTextActionListener(evt):
	Normalization.invalidatePlateAnalysis()
	newVal = float(evt.getActionCommand())
	source = evt.getSource()
	sliderObj = masterObject.getField('maxThreshSlider')
	slider = sliderObj.slider
	minThresh = sliderObj.minThresh
	maxThresh = sliderObj.maxThresh
	step = sliderObj.stepSize

	# compute the position of the slider:
	sVal = int((newVal - minThresh) / step)
	slider.setValue(sVal)

# action listener fired when the histogram type is changed
def ieHistTypeActionListener(evt):
	# get/set the new IE histogram type
	masterObject.setField('ieHistType', evt.getActionCommand())
	vPlateSet = masterObject.getField('vPlateSet')
	IEHistogram(BIN_EDGES)

# action listener fired when one of the virus plate column headers is clicked
def vpColumnSelect(evt, table):
	# get the index of the selected column:
	selectedColIdx = table.getTableHeader().columnAtPoint(evt.getPoint())
	# only re-sort if the selected column has changed:
	# sort the list even if the previous selection was the same
	#if selectedColIdx!=masterObject.getField('vpColumnSortIdx'):
	#	masterObject.setField('vpColumnSortIdx', selectedColIdx)

	# the table is contained within a JViewport, which is contained in the NormTablePane:
	tablePane = table.getParent().getParent()
	tablePane.sortTableContents(selectedColIdx)
	tablePane.table.setRowSelectionInterval(0, 0)
	resetPlateList(0)

# For debugging (not used)
def readHistPoints(inFile, setName):
	dataList = []
	fin = open(inFile, 'r')
	while 1:
		line = fin.readline()
		if not line:
			break

		line = line.strip()

		if not len(line):
			continue

		bin, y = line.split('\t')
		dataList.append([bin, float(y)])

	fin.close()
	return dataList

# For debugging (not used)
def readCsvPointPairs(inFile, setName):
	series = XYSeries(setName)
	fin = open(inFile, 'r')
	while 1:
		line = fin.readline()
		if not line:
			break

		line = line.strip()

		if not len(line):
			continue

		(x, y, name) = line.split(',')
		series.add(float(x), float(y))

	fin.close()
	return series

# For debugging (not used)
def ieValidateAnalysis(dummy):

	print 'Validating InfectionEfficiency analysis----------------------'
	IePanel = masterObject.getField('IePanel')
	print '**IePanel:'
	print '               size: ', IePanel.getSize()
	print '        maximumSize: ', IePanel.getMaximumSize()
	print '      preferredSize: ', IePanel.getPreferredSize()
	print '        minimumSize: ', IePanel.getMinimumSize()
	print ''

	gbPanel = masterObject.getField('gbPanel')
	print ''
	print '      *gbPanel size: ', gbPanel.getSize()
	print ''
	graphBox = masterObject.getField('graphBox')
	print 'graphBox:'
	print '               size: ', graphBox.getSize()
	print '        maximumSize: ', graphBox.getMaximumSize()
	print '      preferredSize: ', graphBox.getPreferredSize()
	print '        minimumSize: ', graphBox.getMinimumSize()
	print ''

	vpPanel = masterObject.getField('vpPanel')
	print '      *vpPanel size: ', vpPanel.getSize()
	print ''
	vpListPanel = masterObject.getField('vpListPanel')
	print 'vpListPanel:'
	print '               size: ', vpListPanel.getSize()
	print '        maximumSize: ', vpListPanel.getMaximumSize()
	print '      preferredSize: ', vpListPanel.getPreferredSize()
	print '        minimumSize: ', vpListPanel.getMinimumSize()
	print ''

	iePanel = masterObject.getField('iePanel')
	print '*histTypePanel size: ', iePanel.getSize()
	print ''

	ieHistTypeBox = masterObject.getField('ieHistTypeBoxPanel')
	print 'ieHistTypeBoxPanel:'
	print '               size: ', ieHistTypeBox.getSize()
	print '        maximumSize: ', ieHistTypeBox.getMaximumSize()
	print '      preferredSize: ', ieHistTypeBox.getPreferredSize()
	print '        minimumSize: ', ieHistTypeBox.getMinimumSize()
	print ''

	hpPanel = masterObject.getField('hpPanel')
	print '    *histPanel size: ', hpPanel.getSize()
	print ''
	histoPanel = masterObject.getField('histoPanel')
	print 'histoPanel:'
	print '               size: ', histoPanel.getSize()
	print '        minimumSize: ', histoPanel.getMinimumSize()
	print '      preferredSize: ', histoPanel.getPreferredSize()
	print '        maximumSize: ', histoPanel.getMaximumSize()
	print ''

	slPanel = masterObject.getField('slPanel')
	print '      *slPanel size: ', slPanel.getSize()
	print ''
	slideBox = masterObject.getField('slideBox')
	print 'slideBox:'
	print '               size: ', slideBox.getSize()
	print '        maximumSize: ', slideBox.getMaximumSize()
	print '      preferredSize: ', slideBox.getPreferredSize()
	print '        minimumSize: ', slideBox.getMinimumSize()
	print ''

	stPanel = masterObject.getField('stPanel')
	print '      *stPanel size: ', stPanel.getSize()
	print ''
	statsBox = masterObject.getField('statsBox')
	print 'statsBox:'
	print '               size: ', statsBox.getSize()
	print '        maximumSize: ', statsBox.getMaximumSize()
	print '      preferredSize: ', statsBox.getPreferredSize()
	print '        minimumSize: ', statsBox.getMinimumSize()
	print ''

# initialize the infection efficiency dotplot
def initializeIEPlot():
	topWindow = masterObject.getField('topWindow')

	# select the 'Infection Efficiency' tab:
	topWindow.selectTab('Infection Efficiency')

	# get the dotplot box and repaint it
	graphBox = masterObject.getField('graphBox')
	g = graphBox.getGraphics()
	graphBox.paint(g)

	# get the infection efficiency data to plot
	ieData = masterObject.getField('ieData')

	# remove any existing data:
	graphBox.clearChart()

	# for each virus plate, create a data set to be displayed in each layer:
	vIePlateGroups = {}

	# list of top-level virus plate IDs:
	vPlateIdList = ieData.keys()
	#vCondList = []	# condition list
	# get the mapping of virus plate IDs to virus plate names
	vpId2NameMap = masterObject.getField('vplateId2VPlateName')

	# initialize maximum variables
	setMax = 0
	maxData = 0
	vbPlateIdList = []
	for vPlateId in vPlateIdList:
		for cond in ieData[vPlateId].keys():
			# plot each condition as a different set of points
			vpName = vpId2NameMap[vPlateId]
			vbId = '%s:%s' % (vpName, cond)
			#print 'vbId: %s' % vbId
			vIePlateGroups.setdefault(vbId, {})
			vbPlateIdList.append(vbId)
			# make a plottable pointset
			(hPointset, cPointset, ePointset, pPointset, typeCount, xMax, yMax) = ieData[vPlateId][cond].getPointSets(cond)
			# get the counts of each type of point
			[nHp, nEm, nCt, nPg] = typeCount

			# add valid hairpins for this plate in blue:
			# valid hairpins are tagged with the virus plate ID/condition...
			graphBox.addPointset(vbId, hPointset, HP_COLOR)
			# empties in red (other points are tagged with the hairpin tag, plus a 
			#  suffix indicating the point type (empty, control, or pgw)
			eName = '%s|empty' % vbId
			graphBox.addPointset(eName, ePointset, EMP_COLOR)
			# controls in green
			cName = '%s|control' % vbId
			graphBox.addPointset(cName, cPointset, CTL_COLOR)
			# pgws in magenta
			pName = '%s|pgw' % vbId
			graphBox.addPointset(pName, pPointset, PGW_COLOR)

			# get max for this plate and for the full plate set:
			maxData = max(xMax, yMax)
			if maxData > setMax:
				setMax = maxData

			# save the pointsets
			vIePlateGroups[vbId]['hPlotPoints'] = hPointset
			vIePlateGroups[vbId]['ePlotPoints'] = ePointset
			vIePlateGroups[vbId]['cPlotPoints'] = cPointset
			vIePlateGroups[vbId]['pPlotPoints'] = pPointset
			vIePlateGroups[vbId]['maxData'] = maxData

	# update sliders and add side-effects after all of the points are added and displayed
	masterObject.setField('maxPuroMinus', maxData)
	masterObject.getField('lowCutoffThreshSlider').updateLimits(0, maxData, 0)

	masterObject.getField('minThreshSlider').addSliderSideEffect(graphBox.setLowerLimitSlope)
	masterObject.getField('lowCutoffThreshSlider').addSliderSideEffect(graphBox.setLowerLimitCutoff)
	masterObject.getField('maxThreshSlider').addSliderSideEffect(graphBox.setUpperLimitSlope)

	# invalidate previously calculated scores if the sliders are moved:
	masterObject.getField('minThreshSlider').addSliderSideEffect(Normalization.invalidatePlateAnalysis)
	masterObject.getField('lowCutoffThreshSlider').addSliderSideEffect(Normalization.invalidatePlateAnalysis)
	masterObject.getField('maxThreshSlider').addSliderSideEffect(Normalization.invalidatePlateAnalysis)

	# update the histogram if the sliders are moved
	vpListBox = masterObject.getField('vpListBox')
	masterObject.getField('minThreshSlider').addSliderSideEffect(IEHistogram, [BIN_EDGES])
	masterObject.getField('lowCutoffThreshSlider').addSliderSideEffect(IEHistogram, [BIN_EDGES])
	masterObject.getField('maxThreshSlider').addSliderSideEffect(IEHistogram, [BIN_EDGES])

	# disable the "Examine all plate" button to start with
	masterObject.getField('resetIEListButton').setEnabled(0)

	# save PointSets:
	masterObject.setField('vIePlateGroups', vIePlateGroups)

	# initialize to all plates visible
	masterObject.setField('visiblePlates', vbPlateIdList)

	# set column alignments:
	masterObject.getField('vpListBox').setColumnHorizontalAlignment(JLabel.CENTER)

	# set the background color of the last two cols to the same as the threshold triangles:
	bgColor = Color.newColor(255, 255, 180)
	masterObject.getField('vpListBox').setColumnBackground(6, bgColor)
	masterObject.getField('vpListBox').setColumnBackground(7, bgColor)

	# compute and display the IE histogram:
	histoPointset = IEHistogram(BIN_EDGES)

	masterObject.setField('allPlates', vbPlateIdList)

	# set the maximum 'visible' x value, and square the axes to that value:
	xMax = graphBox.dataset.getDomainUpperBound(0)
	masterObject.setField('ieXmaxVisible', xMax)

	# reset the axes limits before loading the new data:
	graphBox.xyplot.getDomainAxis().setRange(0, xMax)
	graphBox.xyplot.getRangeAxis().setRange(0, xMax)
	graphBox.addUpperLimitTriangle()
	graphBox.addLowerLimitTriangle()
	graphBox.addUnitSlopeLine()

# action listener fired when row(s) are selected from the virus plate table
def vpListActionListener(evt, eSource):
	# return if still adjusting (mouse still held down):
	if evt.getValueIsAdjusting():
		return

	# get the required items from the masterObject:
	graphBox = masterObject.getField('graphBox')
	allPlates = graphBox.getPointsetMap().keys()

	# get the table model for the virus plate list
	vpTable = masterObject.getField('vpListBoxTable')
	vpListBox = masterObject.getField('vpListBox')
	vpTableModel = vpListBox.tableModel

	# get the list of selected rows:
	vpTableRows = vpTable.getSelectedRows()

	# if any plates are selected, use only the selected plates,
	#   otherwise, use all of them:
	if len(vpTableRows):
		vPlateIdx = vpTable.getSelectedRows()
		# if a subset of rowsis selected , enable the "Examine all plates" button
		masterObject.getField('resetIEListButton').setEnabled(1)
	else:
		vPlateIdx = range(vpTable.getRowCount())
		# if no rows are selected (all virus plates), disable the "Examine all plates" button
		masterObject.getField('resetIEListButton').setEnabled(0)

	# make a list of the selected virus plate IDs:
	vSelect = []
	for i in range(len(vPlateIdx)):
		rowIdx = vPlateIdx[i]
		vPlateName = vpTableModel.getValueAt(rowIdx, 0)   # first column
		screen = vpTableModel.getValueAt(rowIdx, 1)       # second columns
		batch = vpTableModel.getValueAt(rowIdx, 2)        # third columns
		cond = vpTableModel.getValueAt(rowIdx, 3)         # fourth column
		if cond == '':
			cond = 'None'    # no condition specified
		# construct identifier for this virus plate/screen/batch/condition 
		vpName = '%s:%s:%s:%s' % (vPlateName, screen, batch, cond)
		# add to the list of selected plates
		vSelect.append(vpName)

	# get the list of all 'visible' pointsets:
	visibleList = []
	for i in range(len(allPlates)):
		thisPlate = allPlates[i]
		for vPlateId in vSelect:
			if thisPlate.startswith(vPlateId):
				visibleList.append(thisPlate)

	# set the alpha for the selected plates to 1.0, others to a low value:
	skipList = ['lowerLimit', 'upperLimit', 'unitSlopeLine_Limit']
	if len(vpTable.getSelectedRows()) > 0:
		scaleSeries = XYSeriesCollection()
		for i in range(len(allPlates)):
			thisPlate = allPlates[i]
			if skipList.count(thisPlate) > 0:
				continue
			if visibleList.count(thisPlate):
				# if this pointset is in the "visible" list, make it visible
				graphBox.setPointsetAlpha(thisPlate, 1.0)
				scaleSeries.addSeries(graphBox.getPointsetSeries(thisPlate))
			else:
				# ... otherwise, make the points barely visible by setting the alpha to 0.05
				graphBox.setPointsetAlpha(thisPlate, 0.05)      # barely visible

			#graphBox.dumpPointsetInfo(thisPlate)

		# get the maximum value of the visible points
		dataMax = scaleSeries.getDomainUpperBound(0)
		# add a 1% buffer to the axes
		masterObject.setField('ieXmaxVisible', dataMax * 1.01)

	else:
		# no plates are selected (all virus plates are visible)
		for i in range(len(allPlates)):
			thisPlate = allPlates[i]
			if skipList.count(thisPlate) > 0:
				continue
			if visibleList.count(thisPlate):
				graphBox.setPointsetAlpha(thisPlate, 1.0)
			else:
				# AD: it doesn't seem like this should ever happen...
				graphBox.setPointsetAlpha(thisPlate, 0.05)

		# set the x-axis maximum
		masterObject.setField('ieXmaxVisible', graphBox.dataset.getDomainUpperBound(0))

	if masterObject.getField('toggleAxisAutoscale') == 'on':
		# if autoscale is "on" set the maximum x- and y-axis limits to the maximum 
		# visible data value
		graphBox.squareAxes(masterObject.getField('ieXmaxVisible'))
	else:
		# otherwise set the axis limits to the maximum value, regardless of visibility
		graphBox.squareAxes()

	# replot the limit triangles:
	graphBox.showLowerLimitTriangle()
	graphBox.showUpperLimitTriangle()

	# compute and display the IE histogram:
	masterObject.setField('visiblePlates', visibleList)
	histoPointset = IEHistogram(BIN_EDGES)

# "Examine all plates" button action listener
def resetPlateList(event):
	# automatically calls infectionEfficiencyVpSelect_callback, via an ActionEvent
	masterObject.getField('vpListBoxTable').clearSelection()

# toggle the state of the "Axis autoscale..." button
def toggleAxisAutoscale(event):
	axisAutoscale = masterObject.getField('toggleAxisAutoscale')
	button = masterObject.getField('ieAxisScaleButton')
	if axisAutoscale == 'on':
		masterObject.setField('toggleAxisAutoscale', 'off')
		button.setText('Axis autoscale off')
		button.setBackground(Color.newColor(100, 255, 100))
		button.setForeground(Color.BLACK)
		masterObject.getField('graphBox').squareAxes()

	else:
		masterObject.setField('toggleAxisAutoscale', 'on')
		button.setText('Axis autoscale on')
		button.setBackground(Color.newColor(255, 150, 150))
		button.setForeground(Color.BLACK)
		xMax = masterObject.getField('ieXmaxVisible')
		masterObject.getField('graphBox').squareAxes(xMax)

# draw the infection efficiency histogram
def IEHistogram(binEdges, barType=0):
	# get the list of visible plates
	vPlateList = masterObject.getField('visiblePlates')
	# get the full plate set
	plateSet = masterObject.getField('plateSet')

	# get the dot plot object
	graphBox = masterObject.getField('graphBox')
	# and all of the plates with pointsets displayed   
	allPlates = graphBox.getPointsetMap().keys()
	# the histogram chart object
	histoChart = masterObject.getField('histoPanel')

	# IE thresholds
	ieMin = masterObject.getField('IE min:')
	ieMax = masterObject.getField('IE max:')
	xMin = masterObject.getField('Low end cutoff:')

	# pointers to the statistics text boxes
	hpStat = masterObject.getField('hpStat')
	ctStat = masterObject.getField('ctStat')
	emStat = masterObject.getField('emStat')
	pgwStat = masterObject.getField('pgwStat')

	# get the type of histogram:
	histoType = masterObject.getField('ieHistType')
	if histoType == 'Type':
		barType = 1
	elif histoType == 'Non-empty':
		barType = 2
	else:
		barType = 0

	# initialize pass/fail counters:
	hPass = 0
	hFail = 0
	ePass = 0
	eFail = 0
	cPass = 0
	cFail = 0
	pPass = 0
	pFail = 0

	# for counts by virus plate:
	vCounts = {}

	# IE value lists:
	hpValList = []
	emValList = []
	ctValList = []
	pgValList = []

	# make lists of points, grouped by type
	for vbPlateId in allPlates:
		# skip plates with names of the form <plateId>.*
		if vbPlateId.count('|') > 0 or vbPlateId.endswith('Limit'):
			continue

		# initialize counts to 0:
		vCounts.setdefault(vbPlateId, {'h':[0, 0, 0.0], 'e':[0, 0, 0.0], 'c':[0, 0, 0.0], 'p':[0, 0, 0.0]})

		# extract vPlateId and batchId:
		(vPlateName, screenName, batchName, condition) = vbPlateId.split(':')

		# loop over all pointsets
		for plateId in plateSet.keys():
			plate = plateSet[plateId]

			# accumulate stats only for plates with the same vPlateId and batchId and condition:
			# AD: this appears to be a no-op
			if plate.batchName != batchName or plate.virusPlate.plateName != vPlateName:
				continue
			elif plate.get_conditionList().count(condition) == 0:
				#cList = plate.get_conditionList()
				#print 'condition %s not in list (length=%d) '%(condition,len(cList)),cList
				continue

			# invalidate the IE mask for this plate:
			plate.invalidateIE()

			# check if IE data is available for this plate
			if plate.ieData == None:
				print 'Skipping plate (no ieData): %s' % plateId
				continue

			# get the virus plate associated with this plate:
			vPlate = plate.virusPlate

			# examine all wells of this plate
			for row in plate.ieData.ie.keys():
				for col in plate.ieData.ie[row].keys():

					# update counts for 'visible plates'
					ie = plate.ieData.ie[row][col]
					pm = plate.ieData.selMinusData[row][col]
					pp = plate.ieData.selPlusData[row][col]
					if vPlateList.count(vbPlateId):
						if vPlate.well[row][col].type == 1:     # valid hairpin
							hpValList.append(ie)

							# count pass/fail threshold limits:
							if (ie >= ieMin or (pm < xMin and pp < xMin)) and ie <= ieMax:
								hPass += 1
							else:
								hFail += 1

						elif vPlate.well[row][col].type == 0:   # empty
							emValList.append(ie)

							# count pass/fail:
							if (ie >= ieMin or (pm < xMin and pp < xMin)) and ie <= ieMax:
								ePass += 1
							else:
								eFail += 1

						elif vPlate.well[row][col].type == -1:  # control
							ctValList.append(ie)

							# count pass/fail:
							if (ie >= ieMin or (pm < xMin and pp < xMin)) and ie <= ieMax:
								cPass += 1
							else:
								cFail += 1

						elif vPlate.well[row][col].type == -2:  # pgw
							pgValList.append(ie)

							# count pass/fail:
							if (ie >= ieMin or (pm < xMin and pp < xMin)) and ie <= ieMax:
								pPass += 1
							else:
								pFail += 1

					# gather plate-specific counts for all plates:
					if vPlate.well[row][col].type == 1:     # valid hairpin
						vCounts[vbPlateId]['h'][0] += 1
						# count wells removed by thresholding:
						if ie < ieMin or ie > ieMax:
							vCounts[vbPlateId]['h'][1] += 1

					elif vPlate.well[row][col].type == 0:   # empty
						vCounts[vbPlateId]['e'][0] += 1
						# count wells removed by thresholding:
						if ie < ieMin or ie > ieMax:
							vCounts[vbPlateId]['e'][1] += 1

					elif vPlate.well[row][col].type == -1:  # control
						vCounts[vbPlateId]['c'][0] += 1
						# count wells removed by thresholding:
						if ie < ieMin or ie > ieMax:
							vCounts[vbPlateId]['c'][1] += 1

					elif vPlate.well[row][col].type == -2:  # pgw
						vCounts[vbPlateId]['p'][0] += 1
						# count wells removed by thresholding:
						if ie < ieMin or ie > ieMax:
							vCounts[vbPlateId]['p'][1] += 1

	# gather counts:
	nHp = len(hpValList)
	nEm = len(emValList)
	nCt = len(ctValList)
	nPg = len(pgValList)
	nPts = nHp + nEm + nCt

	# print counts:
	if nHp != 0.0:
		hpText = '%d of %d (%6.2f%%)' % (hFail, nHp, 100.0 * hFail / nHp)
	else:
		hpText = '0 of 0 (0.00%%)'
	# update valid hairpin stats
	hpStat.setText(hpText)

	if nCt != 0.0:
		ctText = '%d of %d (%6.2f%%)' % (cFail, nCt, 100.0 * cFail / nCt)
	else:
		ctText = '0 of 0 (0.00%%)'
	# update control hairpin stats
	ctStat.setText(ctText)

	if nEm != 0.0:
		emText = '%d of %d (%6.2f%%)' % (eFail, nEm, 100.0 * eFail / nEm)
	else:
		emText = '0 of 0 (0.00%%)'
	# update EMPTY well stats
	emStat.setText(emText)

	if nPg != 0.0:
		pgwText = '%d of %d (%6.2f%%)' % (pFail, nPg, 100.0 * pFail / nPg)
	else:
		pgwText = '0 of 0 (0.00%%)'
	# update pgw well stats
	pgwStat.setText(pgwText)

	# get the bin counts for the three categories:
	hpBinCount = binCounts(binEdges, hpValList)
	emBinCount = binCounts(binEdges, emValList)
	ctBinCount = binCounts(binEdges, ctValList)
	pgBinCount = binCounts(binEdges, pgValList)

	# create the appropriate pointSet:
	hpPointset = []
	emPointset = []
	ctPointset = []
	pgPointset = []
	if barType == 0:         # normalize to total well count
		if nPts > 0:
			for i in range(len(binEdges)):
				bin = '%3.1f' % binEdges[i]
				hpPointset.append([bin, hpBinCount[i] / float(nPts)])
				emPointset.append([bin, emBinCount[i] / float(nPts)])
				ctPointset.append([bin, ctBinCount[i] / float(nPts)])
				pgPointset.append([bin, pgBinCount[i] / float(nPts)])

			bin = GT_STRING  # wells with infection efficiency greater than 2.0
			hpPointset.append([bin, hpBinCount[i + 1] / float(nPts)])
			emPointset.append([bin, emBinCount[i + 1] / float(nPts)])
			ctPointset.append([bin, ctBinCount[i + 1] / float(nPts)])
			pgPointset.append([bin, pgBinCount[i + 1] / float(nPts)])
	elif barType == 1:       # normalize to category count
		for i in range(len(binEdges)):
			bin = '%3.1f' % binEdges[i]

			if nHp > 0:
				hpPointset.append([bin, hpBinCount[i] / float(nHp)])
			else:
				hpPointset.append([bin, 0])

			if nEm > 0:
				emPointset.append([bin, emBinCount[i] / float(nEm)])
			else:
				emPointset.append([bin, 0])

			if nCt > 0:
				ctPointset.append([bin, ctBinCount[i] / float(nCt)])
			else:
				ctPointset.append([bin, 0])

			if nPg > 0:
				pgPointset.append([bin, pgBinCount[i] / float(nPg)])
			else:
				pgPointset.append([bin, 0])

		bin = GT_STRING
		if nHp > 0:
			hpPointset.append([bin, hpBinCount[i + 1] / float(nHp)])
		else:
			hpPointset.append([bin, 0])

		if nEm > 0:
			emPointset.append([bin, emBinCount[i + 1] / float(nEm)])
		else:
			emPointset.append([bin, 0])

		if nCt > 0:
			ctPointset.append([bin, ctBinCount[i + 1] / float(nCt)])
		else:
			ctPointset.append([bin, 0])

		if nPg > 0:
			pgPointset.append([bin, pgBinCount[i + 1] / float(nPg)])
		else:
			pgPointset.append([bin, 0])

	elif barType == 2:       # normalize to category count, ignore empty wells
		for i in range(len(binEdges)):
			bin = '%3.1f' % binEdges[i]

			if nHp > 0:
				hpPointset.append([bin, hpBinCount[i] / float(nHp)])
			else:
				hpPointset.append([bin, 0])

			emPointset.append([bin, 0])

			if nCt > 0:
				ctPointset.append([bin, ctBinCount[i] / float(nCt)])
			else:
				ctPointset.append([bin, 0])

			if nPg > 0:
				pgPointset.append([bin, pgBinCount[i] / float(nPg)])
			else:
				pgPointset.append([bin, 0])

		bin = GT_STRING
		if nHp > 0:
			hpPointset.append([bin, hpBinCount[i + 1] / float(nHp)])
		else:
			hpPointset.append([bin, 0])

		emPointset.append([bin, 0])

		if nCt > 0:
			ctPointset.append([bin, ctBinCount[i + 1] / float(nCt)])
		else:
			ctPointset.append([bin, 0])

		if nPg > 0:
			pgPointset.append([bin, pgBinCount[i + 1] / float(nPg)])
		else:
			pgPointset.append([bin, 0])

	else:                  # display raw counts
		for i in range(len(binEdges)):
			bin = '%3.1f' % binEdges[i]
			hpPointset.append([bin, float(hpBinCount[i])])
			emPointset.append([bin, float(emBinCount[i])])
			ctPointset.append([bin, float(ctBinCount[i])])
			pgPointset.append([bin, float(pgBinCount[i])])

		#bin = '>%2.1f' % binEdges[i]
		bin = GT_STRING
		hpPointset.append([bin, float(hpBinCount[i + 1])])
		emPointset.append([bin, float(emBinCount[i + 1])])
		ctPointset.append([bin, float(ctBinCount[i + 1])])
		pgPointset.append([bin, float(pgBinCount[i + 1])])

	# clear the chart
	histoChart.resetPointset()
	# then add each of the pointsets to the histogram
	histoChart.addPointset('hairpin', hpPointset, HP_COLOR)
	histoChart.addPointset('empty', emPointset, EMP_COLOR)
	histoChart.addPointset('control', ctPointset, CTL_COLOR)
	histoChart.addPointset('pgw', pgPointset, PGW_COLOR)

	###############################################
	# compute plate-specific percentages of wells removed by thresholding
	# (only for hairpins and empty wells):
	for vPlateId in vCounts.keys():
		hCounts = vCounts[vPlateId]['h']
		if hCounts[0] > 0:
			hCounts[2] = float(hCounts[1]) / hCounts[0]

		eCounts = vCounts[vPlateId]['e']
		if eCounts[0] > 0:
			eCounts[2] = float(eCounts[1]) / eCounts[0]

	# reset the vpListBox table:
	vpListBox = masterObject.getField('vpListBox')
	vpSelModel = vpListBox.table.getSelectionModel()
	vpTable = masterObject.getField('vpListBoxTable')
	vpTableModel = vpListBox.tableModel

	# remove listeners while updating the table:
	# (Otherwise, they will fire every time anything is updated)
	listeners = vpSelModel.getListSelectionListeners()
	for listener in listeners:
		vpSelModel.removeListSelectionListener(listener)

	# update rows with new entries for hp% and empty%
	vpRowN = vpTable.getRowCount()
	#print 'vCounts.keys(): ', vCounts.keys()
	for rowIdx in range(vpRowN):
		vPlateName = vpTableModel.getValueAt(rowIdx, 0)
		screen = vpTableModel.getValueAt(rowIdx, 1)
		batch = vpTableModel.getValueAt(rowIdx, 2)
		cond = vpTableModel.getValueAt(rowIdx, 3)
		if cond == '':
			cond = 'None'
		vpName = '%s:%s:%s:%s' % (vPlateName, screen, batch, cond)
		#print 'vpName: %s' % vpName

		try:
			hPctNum = 100.0 * vCounts[vpName]['h'][2]
			if hPctNum == 100.0:
				hPct = '100.00'
			elif hPctNum >= 10.0:
				hPct = ' %4.2f' % hPctNum
			else:
				hPct = '  %3.2f' % hPctNum

		except:
			# no data
			hPct = '----'

		try:
			ePctNum = 100.0 * vCounts[vpName]['e'][2]
			if ePctNum == 100.0:
				ePct = '100.00'
			elif ePctNum >= 10.0:
				ePct = ' %4.2f' % ePctNum
			else:
				ePct = '  %3.2f' % ePctNum
		except:
			# no data
			ePct = '----'

		vpTableModel.setValueAt(hPct, rowIdx, 6)
		vpTableModel.setValueAt(ePct, rowIdx, 7)

	# replace listeners after updating the table:
	for listener in listeners:
		vpSelModel.addListSelectionListener(listener)

# compute the heights of the histogram bars
def binCounts(binEdges, dataList):
	dataList.sort()
	nBins = len(binEdges)
	binCount = [0] * (nBins + 1)
	# since the values are sorted, we can step through each list and
	# increment the bin counter when a value exceeds the bin edge:
	binIdx = 0
	for i in range(len(dataList)):
		x = dataList[i]
		if x <= binEdges[binIdx]:
			# increment the count for this bin
			binCount[binIdx] += 1
		else:
			# step to the next bin
			binIdx = getNextBinIdx(binEdges, x)
			if binIdx == nBins:
				# get the count for the last bin as the length of the remainder of the list
				binCount[binIdx] += len(dataList[i:])
				break
			else:
				# increment the count for this bin
				binCount[binIdx] += 1

	return binCount

# get the index of the bin that contains this value
def getNextBinIdx(binEdges, x):
	idx = 0
	for bin in binEdges:
		if x < bin:
			break
		idx += 1
	return idx






