/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.rep.impl.networkRestore;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.rep.impl.RepImpl;
import com.sleepycat.je.rep.impl.networkRestore.LogFileFeeder;
import com.sleepycat.je.rep.impl.networkRestore.NetworkBackupStatDefinition;
import com.sleepycat.je.rep.impl.networkRestore.Protocol;
import com.sleepycat.je.rep.impl.node.NameIdPair;
import com.sleepycat.je.rep.utilint.BinaryProtocol;
import com.sleepycat.je.rep.utilint.RepUtils;
import com.sleepycat.je.rep.utilint.ServiceDispatcher;
import com.sleepycat.je.utilint.IntStat;
import com.sleepycat.je.utilint.LoggerUtils;
import com.sleepycat.je.utilint.StatGroup;
import com.sleepycat.je.utilint.VLSN;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.logging.Logger;

public class NetworkBackup {
    private final InetSocketAddress serverAddress;
    private final File envDir;
    private final NameIdPair clientNameId;
    private final boolean retainLogfiles;
    private final VLSN minVLSN;
    final int serverLoadThreshold;
    private final RepImpl repImpl;
    private final FileManager fileManager;
    private Protocol protocol;
    private SocketChannel channel;
    final MessageDigest messageDigest;
    private final StatGroup statistics;
    private final IntStat backupFileCount;
    private final IntStat disposedCount;
    private final IntStat fetchCount;
    private final IntStat skipCount;
    private final Logger logger;
    private CyclicBarrier testBarrier = null;
    private final int receiveBufferSize;
    private static final int SOCKET_TIMEOUT_MS = 10000;
    private static final int DIGEST_RETRIES = 5;

    public NetworkBackup(InetSocketAddress serverSocket, int receiveBufferSize, File envDir, NameIdPair clientNameId, boolean retainLogfiles, int serverLoadThreshold, VLSN minVLSN, RepImpl repImpl, FileManager fileManager) throws IllegalArgumentException {
        this.serverAddress = serverSocket;
        this.receiveBufferSize = receiveBufferSize;
        if (!envDir.exists()) {
            throw new IllegalArgumentException("Environment directory: " + envDir + " not found");
        }
        this.envDir = envDir;
        this.clientNameId = clientNameId;
        this.retainLogfiles = retainLogfiles;
        this.serverLoadThreshold = serverLoadThreshold;
        this.minVLSN = minVLSN;
        this.repImpl = repImpl;
        this.fileManager = fileManager;
        try {
            this.messageDigest = MessageDigest.getInstance("SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            throw EnvironmentFailureException.unexpectedException(e);
        }
        this.logger = LoggerUtils.getLoggerFixedPrefix(this.getClass(), clientNameId.toString(), repImpl);
        this.statistics = new StatGroup("NetworkBackup", "NetworkBackup statistics");
        this.backupFileCount = new IntStat(this.statistics, NetworkBackupStatDefinition.BACKUP_FILE_COUNT);
        this.disposedCount = new IntStat(this.statistics, NetworkBackupStatDefinition.DISPOSED_COUNT);
        this.fetchCount = new IntStat(this.statistics, NetworkBackupStatDefinition.FETCH_COUNT);
        this.skipCount = new IntStat(this.statistics, NetworkBackupStatDefinition.SKIP_COUNT);
    }

    public NetworkBackup(InetSocketAddress serverSocket, File envDir, NameIdPair clientNameId, boolean retainLogfiles, FileManager fileManager) throws DatabaseException {
        this(serverSocket, 0, envDir, clientNameId, retainLogfiles, Integer.MAX_VALUE, VLSN.NULL_VLSN, null, fileManager);
    }

    public StatGroup getStats() {
        StatGroup ret = this.statistics.cloneGroup(false);
        return ret;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String[] execute() throws IOException, DatabaseException, ServiceDispatcher.ServiceConnectFailedException, LoadThresholdExceededException, InsufficientVLSNRangeException {
        String[] stringArray;
        block4: {
            try {
                this.channel = RepUtils.openBlockingChannel(this.serverAddress, true, this.receiveBufferSize, 10000);
                ServiceDispatcher.doServiceHandshake(this.channel, "LogFileFeeder");
                this.protocol = this.checkProtocol(new Protocol(this.clientNameId, 2, this.repImpl));
                this.checkServer();
                String[] fileNames = this.getFileList();
                this.logger.info("Restoring from:" + this.serverAddress + " Allocated network receive buffer size:" + this.channel.socket().getReceiveBufferSize() + "(" + this.receiveBufferSize + ")" + " candidate log file count:" + fileNames.length);
                this.getFiles(fileNames);
                this.cleanup(fileNames);
                assert (this.fileManager.listJDBFiles().length == fileNames.length) : "envDir=" + this.envDir + " list=" + Arrays.asList(this.fileManager.listJDBFiles()) + " fileNames=" + Arrays.asList(fileNames);
                long fileBegin = this.fileManager.getNumFromName(fileNames[0]);
                long fileEnd = this.fileManager.getNumFromName(fileNames[fileNames.length - 1]);
                stringArray = this.fileManager.listFileNames(fileBegin, fileEnd);
                Object var8_5 = null;
                if (this.channel == null) break block4;
            }
            catch (Throwable throwable) {
                Object var8_6 = null;
                if (this.channel != null) {
                    this.channel.socket().close();
                    this.channel.close();
                }
                this.logger.info("Backup file total: " + this.backupFileCount.get() + ".  Files actually fetched: " + this.fetchCount.get() + ".  Files skipped(available locally): " + this.skipCount.get() + ".  Local files renamed/deleted: " + this.disposedCount.get());
                throw throwable;
            }
            this.channel.socket().close();
            this.channel.close();
        }
        this.logger.info("Backup file total: " + this.backupFileCount.get() + ".  Files actually fetched: " + this.fetchCount.get() + ".  Files skipped(available locally): " + this.skipCount.get() + ".  Local files renamed/deleted: " + this.disposedCount.get());
        return stringArray;
    }

    private void checkServer() throws IOException, BinaryProtocol.ProtocolException, LoadThresholdExceededException, InsufficientVLSNRangeException {
        this.protocol.write((BinaryProtocol.Message)new Protocol.FeederInfoReq(this.protocol), this.channel);
        Protocol.FeederInfoResp resp = this.protocol.read(this.channel, Protocol.FeederInfoResp.class);
        if (!(VLSN.NULL_VLSN.equals(this.minVLSN) || resp.getRangeFirst().compareTo(this.minVLSN) <= 0 && resp.getRangeLast().compareTo(this.minVLSN) >= 0)) {
            throw new InsufficientVLSNRangeException(this.minVLSN, resp.getRangeFirst(), resp.getRangeLast());
        }
        if (resp.getActiveFeeders() > this.serverLoadThreshold) {
            LoadThresholdExceededException exception = new LoadThresholdExceededException(this.serverLoadThreshold, resp.getActiveFeeders());
            throw exception;
        }
    }

    private void cleanup(String[] fileNames) throws IOException {
        this.logger.fine("Cleaning up");
        HashSet<String> logFileSet = new HashSet<String>(Arrays.asList(fileNames));
        for (File file : this.fileManager.listJDBFiles()) {
            if (logFileSet.contains(file.getName())) continue;
            this.disposeFile(file);
        }
        StringBuilder logFiles = new StringBuilder();
        for (String string2 : logFileSet) {
            File file;
            file = new File(this.fileManager.getFullFileName(string2));
            if (!file.exists()) {
                throw EnvironmentFailureException.unexpectedState("Missing file: " + file);
            }
            logFiles.append(file.getCanonicalPath()).append(", ");
        }
        String names = logFiles.toString();
        if (names.length() > 0) {
            names = names.substring(0, names.length() - 2);
        }
        this.logger.fine("Log file set: " + names);
    }

    private void getFiles(String[] fileNames) throws IOException, DatabaseException {
        this.logger.info(fileNames.length + " files in backup set");
        block5: for (String fileName : fileNames) {
            File file;
            if (this.testBarrier != null) {
                try {
                    this.testBarrier.await();
                }
                catch (InterruptedException e) {
                }
                catch (BrokenBarrierException e) {
                    throw EnvironmentFailureException.unexpectedException(e);
                }
            }
            if (this.haveFile(file = new File(this.fileManager.getFullFileName(fileName)))) {
                this.logger.info("File: " + file.getCanonicalPath() + " length: " + file.length() + " available with matching SHA1, copy skipped");
                this.skipCount.increment();
                continue;
            }
            for (int i = 0; i < 5; ++i) {
                try {
                    this.getFile(file);
                    this.fetchCount.increment();
                    continue block5;
                }
                catch (DigestException e) {
                    if (i + 1 != 5) continue;
                    throw new IOException("Digest mismatch despite 5 attempts");
                }
            }
        }
        this.protocol.write((BinaryProtocol.Message)new Protocol.Done(this.protocol), this.channel);
    }

    private boolean haveFile(File file, boolean getSHA1) throws IOException, DatabaseException {
        if (!file.exists()) {
            return false;
        }
        Protocol protocol = this.protocol;
        protocol.getClass();
        this.protocol.write((BinaryProtocol.Message)new Protocol.FileInfoReq(protocol, file.getName(), getSHA1), this.channel);
        Protocol.FileInfoResp statResp = this.protocol.read(this.channel, Protocol.FileInfoResp.class);
        long fileLength = file.length();
        if (statResp.getFileLength() != fileLength) {
            return false;
        }
        if (statResp.getDigestSHA1().length == 0) {
            assert (!getSHA1);
            return this.haveFile(file, true);
        }
        byte[] digest = LogFileFeeder.getSHA1Digest(file, file.length()).digest();
        return Arrays.equals(digest, statResp.getDigestSHA1());
    }

    private boolean haveFile(File file) throws IOException, DatabaseException {
        return this.haveFile(file, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void getFile(File file) throws IOException, BinaryProtocol.ProtocolException, DigestException {
        boolean deleted;
        this.logger.fine("Requesting file: " + file);
        Protocol protocol = this.protocol;
        protocol.getClass();
        this.protocol.write((BinaryProtocol.Message)new Protocol.FileReq(protocol, file.getName()), this.channel);
        Protocol.FileStart fileResp = this.protocol.read(this.channel, Protocol.FileStart.class);
        File tmpFile = new File(this.fileManager.getFullFileName(file.getName()) + ".tmp");
        if (tmpFile.exists() && !(deleted = tmpFile.delete())) {
            throw EnvironmentFailureException.unexpectedState("Could not delete file: " + tmpFile);
        }
        ByteBuffer buffer = ByteBuffer.allocateDirect(8192);
        this.messageDigest.reset();
        FileOutputStream fileStream = new FileOutputStream(tmpFile);
        FileChannel fileChannel = fileStream.getChannel();
        try {
            int actualBytes;
            for (long bytes = fileResp.getFileLength(); bytes > 0L; bytes -= (long)actualBytes) {
                int readSize = (int)Math.min(8192L, bytes);
                buffer.clear();
                buffer.limit(readSize);
                actualBytes = this.channel.read(buffer);
                if (actualBytes == -1) {
                    throw new IOException("Premature EOF. Was expecting:" + readSize);
                }
                buffer.flip();
                fileChannel.write(buffer);
                buffer.rewind();
                this.messageDigest.update(buffer);
            }
            this.logger.info(String.format("Fetched log file: %s, size: %,d bytes", file.getName(), fileResp.getFileLength()));
            Object var12_12 = null;
        }
        catch (Throwable throwable) {
            Object var12_13 = null;
            fileStream.close();
            throw throwable;
        }
        fileStream.close();
        Protocol.FileEnd fileEnd = this.protocol.read(this.channel, Protocol.FileEnd.class);
        if (!Arrays.equals(this.messageDigest.digest(), fileEnd.getDigestSHA1())) {
            this.logger.warning("digest mismatch on file: " + file);
            throw new DigestException();
        }
        if (file.exists()) {
            this.disposeObsoleteFiles(file);
        }
        this.logger.fine("Renamed " + tmpFile + " to " + file);
        boolean renamed = tmpFile.renameTo(file);
        if (!renamed) {
            throw EnvironmentFailureException.unexpectedState("Rename from: " + tmpFile + " to " + file + " failed");
        }
        if (!file.setLastModified(fileResp.getLastModifiedTime())) {
            throw EnvironmentFailureException.unexpectedState("File.setlastModifiedTime() for:" + file + " and time " + new Date(fileResp.getLastModifiedTime()) + " failed.");
        }
    }

    private void disposeObsoleteFiles(File startFile) {
        Object[] dirFiles = this.fileManager.listJDBFiles();
        Arrays.sort(dirFiles);
        for (int i = dirFiles.length - 1; i >= 0; --i) {
            Object file = dirFiles[i];
            this.disposeFile((File)file);
            if (startFile.equals(file)) break;
        }
    }

    private void disposeFile(File file) {
        this.disposedCount.increment();
        if (this.retainLogfiles) {
            long fileNumber = this.fileManager.getNumFromName(file.getName());
            boolean renamed = false;
            try {
                renamed = this.fileManager.renameFile(fileNumber, ".bup");
            }
            catch (IOException e) {
                throw EnvironmentFailureException.unexpectedState("Could not rename log file " + file.getPath() + " because of exception: " + e.getMessage());
            }
            if (!renamed) {
                throw EnvironmentFailureException.unexpectedState("Could not rename log file " + file.getPath());
            }
            this.logger.fine("Renamed log file: " + file.getPath());
        } else {
            boolean deleted = file.delete();
            if (!deleted) {
                throw EnvironmentFailureException.unexpectedState("Could not delete log file " + file.getPath());
            }
            this.logger.fine("deleted log file: " + file.getPath());
        }
    }

    private String[] getFileList() throws IOException, BinaryProtocol.ProtocolException {
        this.protocol.write((BinaryProtocol.Message)new Protocol.FileListReq(this.protocol), this.channel);
        Protocol.FileListResp fileListResp = this.protocol.read(this.channel, Protocol.FileListResp.class);
        Object[] fileList = fileListResp.getFileNames();
        Arrays.sort(fileList);
        this.backupFileCount.set(fileList.length);
        return fileList;
    }

    private Protocol checkProtocol(Protocol candidateProtocol) throws IOException, BinaryProtocol.ProtocolException {
        candidateProtocol.write((BinaryProtocol.Message)new BinaryProtocol.ClientVersion(candidateProtocol), this.channel);
        BinaryProtocol.ServerVersion serverVersion = candidateProtocol.read(this.channel, BinaryProtocol.ServerVersion.class);
        if (serverVersion.getVersion() != candidateProtocol.getVersion()) {
            String message = "Server requested protocol version:" + serverVersion.getVersion() + " but the client version is " + candidateProtocol.getVersion();
            this.logger.info(message);
            throw new BinaryProtocol.ProtocolException(message);
        }
        return candidateProtocol;
    }

    public void setTestBarrier(CyclicBarrier testBarrier) {
        this.testBarrier = testBarrier;
    }

    public static class LoadThresholdExceededException
    extends Exception {
        final int threshold;
        final int activeServers;

        LoadThresholdExceededException(int threshold, int activeServers) {
            assert (activeServers > threshold);
            this.threshold = threshold;
            this.activeServers = activeServers;
        }

        public int getActiveServers() {
            return this.activeServers;
        }

        public int getThreshold() {
            return this.threshold;
        }

        public String getMessage() {
            return "Active server threshold: " + this.threshold + " exceeded. " + "Active servers: " + this.activeServers;
        }
    }

    public static class InsufficientVLSNRangeException
    extends Exception {
        private final VLSN minVLSN;
        private final VLSN rangeFirst;
        private final VLSN rangeLast;

        public InsufficientVLSNRangeException(VLSN minVLSN, VLSN rangeFirst, VLSN rangeLast) {
            this.minVLSN = minVLSN;
            this.rangeFirst = rangeFirst;
            this.rangeLast = rangeLast;
        }

        public VLSN getMinVLSN() {
            return this.minVLSN;
        }

        public VLSN getRangeFirst() {
            return this.rangeFirst;
        }

        public String getMessage() {
            return "Insufficient VLSN range. Needed VLSN: " + this.minVLSN + " Available range: " + "[" + this.rangeFirst + ", " + this.rangeLast + "]";
        }
    }

    protected static class DigestException
    extends Exception {
        protected DigestException() {
        }
    }
}

