/*
 * Decompiled with CFR 0.152.
 */
package org.xerial.snappy;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.util.Arrays;
import org.xerial.snappy.Snappy;
import org.xerial.snappy.SnappyFramed;

public final class SnappyFramedInputStream
extends InputStream
implements ReadableByteChannel {
    private final ReadableByteChannel rbc;
    private final ByteBuffer frameHeader;
    private final boolean verifyChecksums;
    private ByteBuffer input;
    private ByteBuffer uncompressedDirect;
    private boolean closed;
    private boolean eof;
    private int valid;
    private int position;
    private byte[] buffer;

    public SnappyFramedInputStream(InputStream in) throws IOException {
        this(in, true);
    }

    public SnappyFramedInputStream(InputStream in, boolean verifyChecksums) throws IOException {
        this(Channels.newChannel(in), verifyChecksums);
    }

    public SnappyFramedInputStream(ReadableByteChannel in) throws IOException {
        this(in, true);
    }

    public SnappyFramedInputStream(ReadableByteChannel in, boolean verifyChecksums) throws IOException {
        if (in == null) {
            throw new NullPointerException("in is null");
        }
        this.rbc = in;
        this.verifyChecksums = verifyChecksums;
        this.allocateBuffersBasedOnSize(65541);
        this.frameHeader = ByteBuffer.allocate(4);
        byte[] expectedHeader = SnappyFramed.HEADER_BYTES;
        byte[] actualHeader = new byte[expectedHeader.length];
        ByteBuffer actualBuffer = ByteBuffer.wrap(actualHeader);
        int read = in.read(actualBuffer);
        if (read < expectedHeader.length) {
            throw new EOFException("encountered EOF while reading stream header");
        }
        if (!Arrays.equals(expectedHeader, actualHeader)) {
            throw new IOException("invalid stream header");
        }
    }

    private void allocateBuffersBasedOnSize(int size2) {
        this.input = ByteBuffer.allocateDirect(size2);
        int maxCompressedLength = Snappy.maxCompressedLength(size2);
        this.uncompressedDirect = ByteBuffer.allocateDirect(maxCompressedLength);
        this.buffer = new byte[maxCompressedLength];
    }

    @Override
    public int read() throws IOException {
        if (this.closed) {
            return -1;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        return this.buffer[this.position++] & 0xFF;
    }

    @Override
    public int read(byte[] output, int offset, int length) throws IOException {
        if (output == null) {
            throw new IllegalArgumentException("output is null");
        }
        if (offset < 0 || length < 0 || offset + length > output.length) {
            throw new IllegalArgumentException("invalid offset [" + offset + "] and length [" + length + ']');
        }
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (length == 0) {
            return 0;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        int size2 = Math.min(length, this.available());
        System.arraycopy(this.buffer, this.position, output, offset, size2);
        this.position += size2;
        return size2;
    }

    @Override
    public int available() throws IOException {
        if (this.closed) {
            return 0;
        }
        return this.valid - this.position;
    }

    @Override
    public boolean isOpen() {
        return !this.closed;
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        if (dst == null) {
            throw new IllegalArgumentException("dst is null");
        }
        if (this.closed) {
            throw new ClosedChannelException();
        }
        if (dst.remaining() == 0) {
            return 0;
        }
        if (!this.ensureBuffer()) {
            return -1;
        }
        int size2 = Math.min(dst.remaining(), this.available());
        dst.put(this.buffer, this.position, size2);
        this.position += size2;
        return size2;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        try {
            this.rbc.close();
        }
        finally {
            if (!this.closed) {
                this.closed = true;
            }
        }
    }

    private boolean ensureBuffer() throws IOException {
        int actualCrc32c;
        if (this.available() > 0) {
            return true;
        }
        if (this.eof) {
            return false;
        }
        if (!this.readBlockHeader()) {
            this.eof = true;
            return false;
        }
        FrameMetaData frameMetaData = this.getFrameMetaData(this.frameHeader);
        if (FrameAction.SKIP == frameMetaData.frameAction) {
            SnappyFramed.skip(this.rbc, frameMetaData.length, ByteBuffer.wrap(this.buffer));
            return this.ensureBuffer();
        }
        if (frameMetaData.length > this.input.capacity()) {
            this.allocateBuffersBasedOnSize(frameMetaData.length);
        }
        this.input.clear();
        this.input.limit(frameMetaData.length);
        int actualRead = SnappyFramed.readBytes(this.rbc, this.input);
        if (actualRead != frameMetaData.length) {
            throw new EOFException("unexpectd EOF when reading frame");
        }
        this.input.flip();
        FrameData frameData = this.getFrameData(this.input);
        if (FrameAction.UNCOMPRESS == frameMetaData.frameAction) {
            this.input.position(frameData.offset);
            int uncompressedLength = Snappy.uncompressedLength(this.input);
            if (uncompressedLength > this.uncompressedDirect.capacity()) {
                this.uncompressedDirect = ByteBuffer.allocateDirect(uncompressedLength);
                this.buffer = new byte[Math.max(this.input.capacity(), uncompressedLength)];
            }
            this.uncompressedDirect.clear();
            this.valid = Snappy.uncompress(this.input, this.uncompressedDirect);
            this.uncompressedDirect.get(this.buffer, 0, this.valid);
            this.position = 0;
        } else {
            this.input.position(frameData.offset);
            this.position = 0;
            this.valid = this.input.remaining();
            this.input.get(this.buffer, 0, this.input.remaining());
        }
        if (this.verifyChecksums && frameData.checkSum != (actualCrc32c = SnappyFramed.maskedCrc32c(this.buffer, this.position, this.valid - this.position))) {
            throw new IOException("Corrupt input: invalid checksum");
        }
        return true;
    }

    private boolean readBlockHeader() throws IOException {
        this.frameHeader.clear();
        int read = SnappyFramed.readBytes(this.rbc, this.frameHeader);
        if (read == -1) {
            return false;
        }
        if (read < this.frameHeader.capacity()) {
            throw new EOFException("encountered EOF while reading block header");
        }
        this.frameHeader.flip();
        return true;
    }

    private FrameMetaData getFrameMetaData(ByteBuffer frameHeader) throws IOException {
        FrameAction frameAction;
        assert (frameHeader.hasArray());
        byte[] frameHeaderArray = frameHeader.array();
        int length = frameHeaderArray[1] & 0xFF;
        length |= (frameHeaderArray[2] & 0xFF) << 8;
        length |= (frameHeaderArray[3] & 0xFF) << 16;
        int minLength = 0;
        int flag = frameHeaderArray[0] & 0xFF;
        switch (flag) {
            case 0: {
                frameAction = FrameAction.UNCOMPRESS;
                minLength = 5;
                break;
            }
            case 1: {
                frameAction = FrameAction.RAW;
                minLength = 5;
                break;
            }
            case 255: {
                if (length != 6) {
                    throw new IOException("stream identifier chunk with invalid length: " + length);
                }
                frameAction = FrameAction.SKIP;
                minLength = 6;
                break;
            }
            default: {
                if (flag <= 127) {
                    throw new IOException("unsupported unskippable chunk: " + Integer.toHexString(flag));
                }
                frameAction = FrameAction.SKIP;
                minLength = 0;
            }
        }
        if (length < minLength) {
            throw new IOException("invalid length: " + length + " for chunk flag: " + Integer.toHexString(flag));
        }
        return new FrameMetaData(frameAction, length);
    }

    private FrameData getFrameData(ByteBuffer content) throws IOException {
        return new FrameData(this.getCrc32c(content), 4);
    }

    private int getCrc32c(ByteBuffer content) {
        int position = content.position();
        return (content.get(position + 3) & 0xFF) << 24 | (content.get(position + 2) & 0xFF) << 16 | (content.get(position + 1) & 0xFF) << 8 | content.get(position) & 0xFF;
    }

    public static final class FrameData {
        final int checkSum;
        final int offset;

        public FrameData(int checkSum, int offset) {
            this.checkSum = checkSum;
            this.offset = offset;
        }
    }

    public static final class FrameMetaData {
        final int length;
        final FrameAction frameAction;

        public FrameMetaData(FrameAction frameAction, int length) {
            this.frameAction = frameAction;
            this.length = length;
        }
    }

    static enum FrameAction {
        RAW,
        SKIP,
        UNCOMPRESS;

    }
}

