/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools;

import htsjdk.samtools.AbstractBAMFileIndex;
import htsjdk.samtools.BAMIndexContent;
import htsjdk.samtools.BAMIndexMetaData;
import htsjdk.samtools.BAMIndexWriter;
import htsjdk.samtools.Bin;
import htsjdk.samtools.BinaryBAMIndexWriter;
import htsjdk.samtools.CRAMIndexer;
import htsjdk.samtools.Chunk;
import htsjdk.samtools.GenomicIndexUtil;
import htsjdk.samtools.LinearIndex;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.ValidationStringency;
import htsjdk.samtools.cram.BAIEntry;
import htsjdk.samtools.cram.build.CramIO;
import htsjdk.samtools.cram.ref.ReferenceContext;
import htsjdk.samtools.cram.structure.AlignmentContext;
import htsjdk.samtools.cram.structure.CompressorCache;
import htsjdk.samtools.cram.structure.Container;
import htsjdk.samtools.cram.structure.CramHeader;
import htsjdk.samtools.seekablestream.SeekableStream;
import htsjdk.samtools.util.BlockCompressedFilePointerUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.RuntimeIOException;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.List;

public class CRAMBAIIndexer
implements CRAMIndexer {
    private final int numReferences;
    private final BAMIndexWriter outputWriter;
    private int currentReference = 0;
    private final CRAMBAIIndexBuilder indexBuilder;
    private final CompressorCache compressorCache = new CompressorCache();

    private CRAMBAIIndexer(File output, SAMFileHeader fileHeader) {
        if (fileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException("CRAM file must be coordinate-sorted for indexing.");
        }
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new CRAMBAIIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    public CRAMBAIIndexer(OutputStream output, SAMFileHeader fileHeader) {
        if (fileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException("CRAM file mut be coordinate-sorted for indexing.");
        }
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new CRAMBAIIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    @Override
    public void processContainer(Container container, ValidationStringency validationStringency) {
        container.getBAIEntries(this.compressorCache).forEach(b -> this.processBAIEntry((BAIEntry)b));
    }

    public final void processBAIEntry(BAIEntry baiEntry) {
        ReferenceContext entryContext = baiEntry.getReferenceContext();
        if (entryContext.isMultiRef()) {
            throw new SAMException("Expecting a single reference or unmapped slice.");
        }
        if (entryContext.isMappedSingleRef()) {
            int reference = entryContext.getReferenceSequenceID();
            if (reference != this.currentReference) {
                this.advanceToReference(reference);
            }
            if (reference != this.currentReference) {
                throw new SAMException(String.format("Unexpected reference %s when constructing index for reference %d for slice", reference, this.currentReference));
            }
        }
        this.indexBuilder.recordBAIEntryIndexMetadata(baiEntry);
        if (entryContext.isMappedSingleRef()) {
            this.indexBuilder.processBAIEntry(baiEntry);
        }
    }

    @Override
    public void finish() {
        this.advanceToReference(this.numReferences);
        this.outputWriter.writeNoCoordinateRecordCount(this.indexBuilder.getNoCoordinateRecordCount());
        this.outputWriter.close();
    }

    private void advanceToReference(int nextReference) {
        while (this.currentReference < nextReference) {
            BAMIndexContent content = this.indexBuilder.processCurrentReference();
            this.outputWriter.writeReference(content);
            ++this.currentReference;
            this.indexBuilder.startNewReference();
        }
    }

    public static void createIndex(SeekableStream stream, File output, Log log, ValidationStringency validationStringency) {
        CramHeader cramHeader = CramIO.readCramHeader(stream);
        SAMFileHeader samFileHeader = Container.readSAMFileHeaderContainer(cramHeader.getCRAMVersion(), stream, null);
        if (samFileHeader.getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
            throw new SAMException(String.format("Input must be coordinate sorted (found %s) to create an index.", new Object[]{samFileHeader.getSortOrder()}));
        }
        CRAMBAIIndexer indexer = new CRAMBAIIndexer(output, samFileHeader);
        Container container = null;
        ProgressLogger progressLogger = new ProgressLogger(log, 1, "indexed", "slices");
        do {
            String sequenceName;
            try {
                container = new Container(cramHeader.getCRAMVersion(), stream, stream.position());
            }
            catch (IOException e) {
                throw new RuntimeIOException("error getting stream position", e);
            }
            if (container == null || container.isEOF()) break;
            indexer.processContainer(container, validationStringency);
            if (null == log) continue;
            AlignmentContext alignmentContext = container.getAlignmentContext();
            ReferenceContext containerReferenceContext = alignmentContext.getReferenceContext();
            switch (containerReferenceContext.getType()) {
                case UNMAPPED_UNPLACED_TYPE: {
                    sequenceName = "?";
                    break;
                }
                case MULTIPLE_REFERENCE_TYPE: {
                    sequenceName = "???";
                    break;
                }
                default: {
                    sequenceName = samFileHeader.getSequence(containerReferenceContext.getReferenceSequenceID()).getSequenceName();
                }
            }
            progressLogger.record(sequenceName, alignmentContext.getAlignmentStart());
        } while (!container.isEOF());
        indexer.finish();
    }

    private class CRAMBAIIndexBuilder {
        private final SAMFileHeader bamHeader;
        private Bin[] bins;
        private int binsSeen = 0;
        private final long[] index = new long[LinearIndex.MAX_LINEAR_INDEX_SIZE];
        private int largestIndexSeen = -1;
        private final BAMIndexMetaData indexStats = new BAMIndexMetaData();

        private CRAMBAIIndexBuilder(SAMFileHeader header) {
            this.bamHeader = header;
        }

        private SAMFileHeader getBamHeader() {
            return this.bamHeader;
        }

        private void recordBAIEntryIndexMetadata(BAIEntry baiEntry) {
            this.indexStats.recordMetaData(baiEntry);
        }

        private int computeIndexingBin(BAIEntry baiEntry) {
            int alignmentStart = baiEntry.getAlignmentStart() - 1;
            int alignmentEnd = baiEntry.getAlignmentStart() + baiEntry.getAlignmentSpan() - 1;
            if (alignmentEnd <= alignmentStart) {
                alignmentEnd = alignmentStart + 1;
            }
            return GenomicIndexUtil.regionToBin(alignmentStart, alignmentEnd);
        }

        private void processBAIEntry(BAIEntry baiEntry) {
            Bin bin;
            ReferenceContext sliceContext = baiEntry.getReferenceContext();
            if (!sliceContext.isMappedSingleRef()) {
                return;
            }
            int reference = sliceContext.getReferenceSequenceID();
            if (reference != CRAMBAIIndexer.this.currentReference) {
                throw new SAMException(String.format("Unexpected reference %s when constructing index for reference %d for slice", reference, CRAMBAIIndexer.this.currentReference));
            }
            int binNum = this.computeIndexingBin(baiEntry);
            if (this.bins == null) {
                SAMSequenceRecord seq2 = this.bamHeader.getSequence(reference);
                this.bins = seq2 == null ? new Bin[37451] : new Bin[AbstractBAMFileIndex.getMaxBinNumberForSequenceLength(seq2.getSequenceLength()) + 1];
            }
            if (this.bins[binNum] != null) {
                bin = this.bins[binNum];
            } else {
                this.bins[binNum] = bin = new Bin(reference, binNum);
                ++this.binsSeen;
            }
            long chunkStart = baiEntry.getContainerStartByteOffset() << 16 | (long)baiEntry.getLandmarkIndex();
            long chunkEnd = (baiEntry.getContainerStartByteOffset() << 16 | (long)baiEntry.getLandmarkIndex()) + 1L;
            Chunk newChunk = new Chunk(chunkStart, chunkEnd);
            List<Chunk> oldChunks = bin.getChunkList();
            if (!bin.containsChunks()) {
                bin.addInitialChunk(newChunk);
            } else {
                Chunk lastChunk = bin.getLastChunk();
                if (BlockCompressedFilePointerUtil.areInSameOrAdjacentBlocks(lastChunk.getChunkEnd(), chunkStart)) {
                    lastChunk.setChunkEnd(chunkEnd);
                } else {
                    oldChunks.add(newChunk);
                    bin.setLastChunk(newChunk);
                }
            }
            int alignmentStart = baiEntry.getAlignmentStart();
            int alignmentEnd = baiEntry.getAlignmentStart() + baiEntry.getAlignmentSpan();
            int startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart);
            int endWindow = alignmentEnd == 0 ? (startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart - 1)) : LinearIndex.convertToLinearIndexOffset(alignmentEnd);
            if (endWindow > this.largestIndexSeen) {
                this.largestIndexSeen = endWindow;
            }
            for (int win = startWindow; win <= endWindow; ++win) {
                if (this.index[win] != 0L && chunkStart >= this.index[win]) continue;
                this.index[win] = chunkStart;
            }
        }

        private BAMIndexContent processCurrentReference() {
            if (this.binsSeen == 0) {
                return null;
            }
            long[] newIndex = new long[this.largestIndexSeen + 1];
            long lastNonZeroOffset = 0L;
            for (int i = 0; i <= this.largestIndexSeen; ++i) {
                if (this.index[i] == 0L) {
                    this.index[i] = lastNonZeroOffset;
                } else {
                    lastNonZeroOffset = this.index[i];
                }
                newIndex[i] = this.index[i];
            }
            LinearIndex linearIndex = new LinearIndex(CRAMBAIIndexer.this.currentReference, 0, newIndex);
            return new BAMIndexContent(CRAMBAIIndexer.this.currentReference, this.bins, this.binsSeen, this.indexStats, linearIndex);
        }

        private long getNoCoordinateRecordCount() {
            return this.indexStats.getNoCoordinateRecordCount();
        }

        private void startNewReference() {
            this.bins = null;
            if (this.binsSeen > 0) {
                Arrays.fill(this.index, 0L);
            }
            this.binsSeen = 0;
            this.largestIndexSeen = -1;
            this.indexStats.newReference();
        }
    }
}

