/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.cram.structure;

import htsjdk.samtools.cram.CRAMException;
import htsjdk.samtools.cram.compression.ExternalCompressor;
import htsjdk.samtools.cram.compression.rans.RANS;
import htsjdk.samtools.cram.encoding.external.ByteArrayStopEncoding;
import htsjdk.samtools.cram.encoding.external.ExternalByteEncoding;
import htsjdk.samtools.cram.encoding.external.ExternalEncoding;
import htsjdk.samtools.cram.encoding.external.ExternalIntegerEncoding;
import htsjdk.samtools.cram.encoding.external.ExternalLongEncoding;
import htsjdk.samtools.cram.io.ITF8;
import htsjdk.samtools.cram.io.InputStreamUtils;
import htsjdk.samtools.cram.structure.CRAMEncodingStrategy;
import htsjdk.samtools.cram.structure.CompressorCache;
import htsjdk.samtools.cram.structure.DataSeries;
import htsjdk.samtools.cram.structure.EncodingDescriptor;
import htsjdk.samtools.cram.structure.EncodingID;
import htsjdk.samtools.cram.structure.block.Block;
import htsjdk.samtools.cram.structure.block.BlockCompressionMethod;
import htsjdk.samtools.util.Log;
import htsjdk.utils.ValidationUtils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

public class CompressionHeaderEncodingMap {
    public static final Set<DataSeries> DATASERIES_NOT_READ_BY_HTSJDK = Collections.unmodifiableSet(new LinkedHashSet<DataSeries>(){
        {
            this.add(DataSeries.TC_TagCount);
            this.add(DataSeries.TN_TagNameAndType);
        }
    });
    private static final Log LOG = Log.getInstance(CompressionHeaderEncodingMap.class);
    private Map<DataSeries, EncodingDescriptor> encodingMap = new TreeMap<DataSeries, EncodingDescriptor>();
    private final Map<Integer, ExternalCompressor> externalCompressors = new TreeMap<Integer, ExternalCompressor>();
    private final CompressorCache compressorCache = new CompressorCache();

    public CompressionHeaderEncodingMap(CRAMEncodingStrategy encodingStrategy) {
        ValidationUtils.nonNull(encodingStrategy, "An encoding strategy must be provided");
        ValidationUtils.validateArg(encodingStrategy.getCustomCompressionHeaderEncodingMap() == null, "A custom compression map cannot be used with this constructor");
        this.putExternalRansOrderZeroEncoding(DataSeries.AP_AlignmentPositionOffset);
        this.putExternalRansOrderOneEncoding(DataSeries.BA_Base);
        this.putExternalRansOrderOneEncoding(DataSeries.BF_BitFlags);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.BS_BaseSubstitutionCode);
        this.putExternalRansOrderOneEncoding(DataSeries.CF_CompressionBitFlags);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.DL_DeletionLength);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.FC_FeatureCode);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.FN_NumberOfReadFeatures);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.FP_FeaturePosition);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.HC_HardClip);
        this.putExternalByteArrayStopTabGzipEncoding(encodingStrategy, DataSeries.IN_Insertion);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.MF_MateBitFlags);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.MQ_MappingQualityScore);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.NF_RecordsToNextFragment);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.NP_NextFragmentAlignmentStart);
        this.putExternalRansOrderOneEncoding(DataSeries.NS_NextFragmentReferenceSequenceID);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.PD_padding);
        this.putExternalRansOrderOneEncoding(DataSeries.QS_QualityScore);
        this.putExternalRansOrderOneEncoding(DataSeries.RG_ReadGroup);
        this.putExternalRansOrderZeroEncoding(DataSeries.RI_RefId);
        this.putExternalRansOrderOneEncoding(DataSeries.RL_ReadLength);
        this.putExternalByteArrayStopTabGzipEncoding(encodingStrategy, DataSeries.RN_ReadName);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.RS_RefSkip);
        this.putExternalByteArrayStopTabGzipEncoding(encodingStrategy, DataSeries.SC_SoftClip);
        this.putExternalGzipEncoding(encodingStrategy, DataSeries.TL_TagIdList);
        this.putExternalRansOrderOneEncoding(DataSeries.TS_InsertSize);
    }

    public CompressionHeaderEncodingMap(InputStream inputStream) {
        int byteSize = ITF8.readUnsignedITF8(inputStream);
        byte[] bytes = new byte[byteSize];
        InputStreamUtils.readFully(inputStream, bytes, 0, bytes.length);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        int mapSize = ITF8.readUnsignedITF8(buffer);
        for (int i = 0; i < mapSize; ++i) {
            String dataSeriesAbbreviation = new String(new byte[]{buffer.get(), buffer.get()});
            DataSeries dataSeries = DataSeries.byCanonicalName(dataSeriesAbbreviation);
            EncodingID id = EncodingID.values()[buffer.get()];
            int paramLen = ITF8.readUnsignedITF8(buffer);
            byte[] paramBytes = new byte[paramLen];
            buffer.get(paramBytes);
            if (DATASERIES_NOT_READ_BY_HTSJDK.contains((Object)dataSeries)) {
                LOG.warn("Ignoring obsolete CRAM dataseries: " + dataSeries.getCanonicalName());
                continue;
            }
            this.encodingMap.put(dataSeries, new EncodingDescriptor(id, paramBytes));
        }
    }

    public void putTagBlockCompression(int tagId, ExternalCompressor compressor) {
        ValidationUtils.validateArg(Arrays.asList(DataSeries.values()).stream().noneMatch(ds -> ds.getExternalBlockContentId() == tagId), String.format("tagID %d overlaps with data series content ID", tagId));
        this.externalCompressors.put(tagId, compressor);
    }

    public EncodingDescriptor getEncodingDescriptorForDataSeries(DataSeries dataSeries) {
        return this.encodingMap.get((Object)dataSeries);
    }

    public List<Integer> getExternalIDs() {
        return new ArrayList<Integer>(this.externalCompressors.keySet());
    }

    public Block createCompressedBlockForStream(Integer contentId, ByteArrayOutputStream outputStream) {
        ExternalCompressor compressor = this.externalCompressors.get(contentId);
        byte[] rawContent = outputStream.toByteArray();
        return Block.createExternalBlock(compressor.getMethod(), contentId, compressor.compress(rawContent), rawContent.length);
    }

    public void write(OutputStream outputStream) throws IOException {
        int size = 0;
        for (Object dataSeries : this.encodingMap.keySet()) {
            if (this.encodingMap.get(dataSeries).getEncodingID() == EncodingID.NULL) continue;
            ++size;
        }
        ByteBuffer mapBuffer = ByteBuffer.allocate(102400);
        ITF8.writeUnsignedITF8(size, mapBuffer);
        for (DataSeries dataSeries : this.encodingMap.keySet()) {
            if (this.encodingMap.get((Object)dataSeries).getEncodingID() == EncodingID.NULL) continue;
            String dataSeriesAbbreviation = dataSeries.getCanonicalName();
            mapBuffer.put((byte)dataSeriesAbbreviation.charAt(0));
            mapBuffer.put((byte)dataSeriesAbbreviation.charAt(1));
            EncodingDescriptor params = this.encodingMap.get((Object)dataSeries);
            mapBuffer.put((byte)(0xFF & params.getEncodingID().getId()));
            ITF8.writeUnsignedITF8(params.getEncodingParameters().length, mapBuffer);
            mapBuffer.put(params.getEncodingParameters());
        }
        mapBuffer.flip();
        byte[] mapBytes = new byte[mapBuffer.limit()];
        mapBuffer.get(mapBytes);
        ITF8.writeUnsignedITF8(mapBytes.length, outputStream);
        outputStream.write(mapBytes);
    }

    public ExternalCompressor getBestExternalCompressor(byte[] data, CRAMEncodingStrategy encodingStrategy) {
        ExternalCompressor rans1;
        int rans1Len;
        ExternalCompressor rans0;
        int rans0Len;
        ExternalCompressor gzip = this.compressorCache.getCompressorForMethod(BlockCompressionMethod.GZIP, encodingStrategy.getGZIPCompressionLevel());
        int gzipLen = gzip.compress(data).length;
        int minLen = Math.min(gzipLen, Math.min(rans0Len = (rans0 = this.compressorCache.getCompressorForMethod(BlockCompressionMethod.RANS, RANS.ORDER.ZERO.ordinal())).compress(data).length, rans1Len = (rans1 = this.compressorCache.getCompressorForMethod(BlockCompressionMethod.RANS, RANS.ORDER.ONE.ordinal())).compress(data).length));
        if (minLen == rans0Len) {
            return rans0;
        }
        if (minLen == rans1Len) {
            return rans1;
        }
        return gzip;
    }

    void putExternalEncoding(DataSeries dataSeries, ExternalCompressor compressor) {
        ExternalEncoding cramEncoding;
        int blockContentID = dataSeries.getExternalBlockContentId();
        switch (dataSeries.getType()) {
            case BYTE: {
                cramEncoding = new ExternalByteEncoding(blockContentID);
                break;
            }
            case INT: {
                cramEncoding = new ExternalIntegerEncoding(blockContentID);
                break;
            }
            case LONG: {
                cramEncoding = new ExternalLongEncoding(blockContentID);
                break;
            }
            case BYTE_ARRAY: {
                cramEncoding = new ExternalByteEncoding(blockContentID);
                break;
            }
            default: {
                throw new CRAMException("Unknown data series value type");
            }
        }
        this.putExternalEncoding(dataSeries, cramEncoding.toEncodingDescriptor(), compressor);
    }

    void putCoreEncoding(DataSeries dataSeries, EncodingDescriptor encodingDescriptor) {
        ValidationUtils.validateArg(!encodingDescriptor.getEncodingID().isExternalEncoding(), "Attempt to use an external encoding as a core encoding");
        if (this.externalCompressors.containsKey(dataSeries.getExternalBlockContentId())) {
            this.externalCompressors.remove(dataSeries.getExternalBlockContentId());
        }
        this.putEncoding(dataSeries, encodingDescriptor);
    }

    private void putEncoding(DataSeries dataSeries, EncodingDescriptor encodingDescriptor) {
        this.encodingMap.put(dataSeries, encodingDescriptor);
    }

    public void putExternalEncoding(DataSeries dataSeries, EncodingDescriptor encodingDescriptor, ExternalCompressor compressor) {
        ValidationUtils.validateArg(encodingDescriptor.getEncodingID().isExternalEncoding(), "Attempt to use an external encoding as a core encoding");
        this.putEncoding(dataSeries, encodingDescriptor);
        this.externalCompressors.put(dataSeries.getExternalBlockContentId(), compressor);
    }

    private void putExternalByteArrayStopTabGzipEncoding(CRAMEncodingStrategy encodingStrategy, DataSeries dataSeries) {
        this.putExternalEncoding(dataSeries, new ByteArrayStopEncoding(9, dataSeries.getExternalBlockContentId()).toEncodingDescriptor(), this.compressorCache.getCompressorForMethod(BlockCompressionMethod.GZIP, encodingStrategy.getGZIPCompressionLevel()));
    }

    private void putExternalGzipEncoding(CRAMEncodingStrategy encodingStrategy, DataSeries dataSeries) {
        this.putExternalEncoding(dataSeries, this.compressorCache.getCompressorForMethod(BlockCompressionMethod.GZIP, encodingStrategy.getGZIPCompressionLevel()));
    }

    private void putExternalRansOrderOneEncoding(DataSeries dataSeries) {
        this.putExternalEncoding(dataSeries, this.compressorCache.getCompressorForMethod(BlockCompressionMethod.RANS, RANS.ORDER.ONE.ordinal()));
    }

    private void putExternalRansOrderZeroEncoding(DataSeries dataSeries) {
        this.putExternalEncoding(dataSeries, this.compressorCache.getCompressorForMethod(BlockCompressionMethod.RANS, RANS.ORDER.ZERO.ordinal()));
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        CompressionHeaderEncodingMap that = (CompressionHeaderEncodingMap)o;
        if (!this.encodingMap.equals(that.encodingMap)) {
            return false;
        }
        return this.externalCompressors.equals(that.externalCompressors);
    }

    public int hashCode() {
        int result = this.encodingMap.hashCode();
        result = 31 * result + this.externalCompressors.hashCode();
        return result;
    }
}

