/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.reference;

import htsjdk.samtools.Defaults;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.AbstractFastaSequenceFile;
import htsjdk.samtools.reference.FastaSequenceIndex;
import htsjdk.samtools.reference.FastaSequenceIndexEntry;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.reference.ReferenceSequenceFileFactory;
import htsjdk.samtools.util.IOUtil;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Iterator;

abstract class AbstractIndexedFastaSequenceFile
extends AbstractFastaSequenceFile {
    private final FastaSequenceIndex index;
    private Iterator<FastaSequenceIndexEntry> indexIterator;

    protected AbstractIndexedFastaSequenceFile(Path path) throws FileNotFoundException {
        this(path, new FastaSequenceIndex(AbstractIndexedFastaSequenceFile.findRequiredFastaIndexFile(path)));
    }

    protected AbstractIndexedFastaSequenceFile(Path path, FastaSequenceIndex index) {
        super(path);
        if (index == null) {
            throw new IllegalArgumentException("Null index for fasta " + path);
        }
        this.index = index;
        IOUtil.assertFileIsReadable(path);
        this.reset();
        if (this.getSequenceDictionary() != null) {
            AbstractIndexedFastaSequenceFile.sanityCheckDictionaryAgainstIndex(path.toAbsolutePath().toString(), this.getSequenceDictionary(), index);
        }
    }

    protected AbstractIndexedFastaSequenceFile(String source2, FastaSequenceIndex index, SAMSequenceDictionary dictionary) {
        super(null, source2, dictionary);
        this.index = index;
        this.reset();
    }

    protected static Path findRequiredFastaIndexFile(Path fastaFile) throws FileNotFoundException {
        Path ret = AbstractIndexedFastaSequenceFile.findFastaIndex(fastaFile);
        if (ret == null) {
            throw new FileNotFoundException(ReferenceSequenceFileFactory.getFastaIndexFileName(fastaFile) + " not found.");
        }
        return ret;
    }

    protected static Path findFastaIndex(Path fastaFile) {
        Path indexFile = ReferenceSequenceFileFactory.getFastaIndexFileName(fastaFile);
        if (!Files.exists(indexFile, new LinkOption[0])) {
            return null;
        }
        return indexFile;
    }

    protected static void sanityCheckDictionaryAgainstIndex(String fastaFile, SAMSequenceDictionary sequenceDictionary, FastaSequenceIndex index) {
        if (sequenceDictionary.getSequences().size() != index.size()) {
            throw new SAMException("Sequence dictionary and index contain different numbers of contigs");
        }
        Iterator<SAMSequenceRecord> sequenceIterator = sequenceDictionary.getSequences().iterator();
        Iterator<FastaSequenceIndexEntry> indexIterator = index.iterator();
        while (sequenceIterator.hasNext() && indexIterator.hasNext()) {
            SAMSequenceRecord sequenceEntry = sequenceIterator.next();
            FastaSequenceIndexEntry indexEntry = indexIterator.next();
            if (!sequenceEntry.getSequenceName().equals(indexEntry.getContig())) {
                throw new SAMException(String.format("Mismatch between sequence dictionary fasta index for %s, sequence '%s' != '%s'.", fastaFile, sequenceEntry.getSequenceName(), indexEntry.getContig()));
            }
            if ((long)sequenceEntry.getSequenceLength() == indexEntry.getSize()) continue;
            throw new SAMException("Index length does not match dictionary length for contig: " + sequenceEntry.getSequenceName());
        }
    }

    public FastaSequenceIndex getIndex() {
        return this.index;
    }

    @Override
    public ReferenceSequence nextSequence() {
        if (!this.indexIterator.hasNext()) {
            return null;
        }
        return this.getSequence(this.indexIterator.next().getContig());
    }

    @Override
    public void reset() {
        this.indexIterator = this.index.iterator();
    }

    @Override
    public final boolean isIndexed() {
        return true;
    }

    @Override
    public ReferenceSequence getSequence(String contig) {
        return this.getSubsequenceAt(contig, 1L, (int)this.index.getIndexEntry(contig).getSize());
    }

    @Override
    public ReferenceSequence getSubsequenceAt(String contig, long start, long stop) {
        if (start > stop + 1L) {
            throw new SAMException(String.format("Malformed query; start point %d lies after end point %d", start, stop));
        }
        FastaSequenceIndexEntry indexEntry = this.getIndex().getIndexEntry(contig);
        if (stop > indexEntry.getSize()) {
            throw new SAMException("Query asks for data past end of contig");
        }
        int length = (int)(stop - start + 1L);
        byte[] target = new byte[length];
        ByteBuffer targetBuffer = ByteBuffer.wrap(target);
        int basesPerLine = indexEntry.getBasesPerLine();
        int bytesPerLine = indexEntry.getBytesPerLine();
        int terminatorLength = bytesPerLine - basesPerLine;
        long startOffset = (start - 1L) / (long)basesPerLine * (long)bytesPerLine + (start - 1L) % (long)basesPerLine;
        long minBufferSize = Math.min((long)Defaults.NON_ZERO_BUFFER_SIZE, (long)(length / basesPerLine + 2) * (long)bytesPerLine);
        if (minBufferSize > Integer.MAX_VALUE) {
            throw new SAMException("Buffer is too large: " + minBufferSize);
        }
        ByteBuffer channelBuffer = ByteBuffer.allocate((int)minBufferSize);
        while (targetBuffer.position() < length) {
            startOffset += (long)Math.max((int)(startOffset % (long)bytesPerLine - (long)basesPerLine + 1L), 0);
            try {
                startOffset += (long)this.readFromPosition(channelBuffer, indexEntry.getLocation() + startOffset);
            }
            catch (IOException ex) {
                throw new SAMException("Unable to load " + contig + "(" + start + ", " + stop + ") from " + this.getSource(), ex);
            }
            channelBuffer.flip();
            int positionInContig = (int)start - 1 + targetBuffer.position();
            int nextBaseSpan = Math.min(basesPerLine - positionInContig % basesPerLine, length - targetBuffer.position());
            int bytesToTransfer = Math.min(nextBaseSpan, channelBuffer.capacity());
            channelBuffer.limit(channelBuffer.position() + bytesToTransfer);
            while (channelBuffer.hasRemaining()) {
                targetBuffer.put(channelBuffer);
                bytesToTransfer = Math.min(basesPerLine, length - targetBuffer.position());
                channelBuffer.limit(Math.min(channelBuffer.position() + bytesToTransfer + terminatorLength, channelBuffer.capacity()));
                channelBuffer.position(Math.min(channelBuffer.position() + terminatorLength, channelBuffer.capacity()));
            }
            channelBuffer.flip();
        }
        return new ReferenceSequence(contig, indexEntry.getSequenceIndex(), target);
    }

    protected abstract int readFromPosition(ByteBuffer var1, long var2) throws IOException;
}

