from javawrap import Color, GridBagConstraints, Font
from javawrap.Ellipse2D import EllipseDouble
from javawrap.Line2D import LineDouble
import MainHandle
import java.awt.BasicStroke as BasicStroke
import java.awt.Dimension as Dimension
import java.awt.GridBagLayout
import java.awt.GridBagLayout as GridBagLayout
import java.awt.GridLayout as GridLayout
import java.awt.Insets as Insets
import java.awt.Polygon as Polygon
import java.awt.event.MouseListener as MouseListener
import java.lang.Object as Object
import java.text.NumberFormat as NumberFormat
import javax.swing as swing
import javax.swing.BorderFactory as BorderFactory
import javax.swing.Box as Box
import javax.swing.BoxLayout as BoxLayout
import javax.swing.ButtonGroup as ButtonGroup
import javax.swing.JButton as JButton
import javax.swing.JFileChooser as JFileChooser
import javax.swing.JFrame as JFrame
import javax.swing.JLabel as JLabel
import javax.swing.JList as JList
import javax.swing.JOptionPane as JOptionPane
import javax.swing.JPanel as JPanel
import javax.swing.JProgressBar as JProgressBar
import javax.swing.JRadioButton as JRadioButton
import javax.swing.JScrollPane as JScrollPane
import javax.swing.JSlider as JSlider
import javax.swing.JTabbedPane as JTabbedPane
import javax.swing.JTable as JTable
import javax.swing.JTextField as JTextField
import javax.swing.ListSelectionModel as ListSelectionModel
import javax.swing.border.TitledBorder as TitledBorder
import javax.swing.event.ChangeEvent as ChangeEvent
import javax.swing.event.ChangeListener as ChangeListener
import javax.swing.event.TableModelListener as TableModelListener
import javax.swing.filechooser.FileFilter as FileFilter
import javax.swing.table.DefaultTableCellRenderer as DefaultTableCellRenderer
import javax.swing.table.DefaultTableColumnModel as DefaultTableColumnModel
import javax.swing.table.DefaultTableModel as DefaultTableModel
import javax.swing.table.TableCellRenderer as TableCellRenderer
import javax.swing.text.NumberFormatter as NumberFormatter
import math
import os.path
import codecs

masterObject = MainHandle.masterObject

N_THRESH_STEP = 100
ALPHABET = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
			'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z']

ROWLIST = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P']
COLLIST = ['01', '02', '03', '04', '05', '06', '07', '08', '09', '10', '11', '12',
		   '13', '14', '15', '16', '17', '18', '19', '20', '21', '22', '23', '24']
COLOR_UNDEFINED = Color.LIGHT_GRAY
#HIGH_COLOR = Color.LIGHT_GRAY
HIGH_COLOR = Color.newColor(0, 255, 0)
#LOW_COLOR = Color.LIGHT_GRAY
LOW_COLOR = Color.newColor(255, 0, 0)


class MainWindow(JFrame):
	# MainWindow is the top-level container for all of the tabs of RNAeyes
	def __init__(self, title, conn, size=[220, 100]):
		# features attached to top-level display:
		self.conn = conn	  # DB connection
		self.prefSize = Dimension(size[0], size[1])
		self.tabMap = {}	  # map/dictionary of tabs

		# initialize the main window:
		JFrame.__init__(self, title=title) #@UndefinedVariable
		self.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE)
		self.contentPane.layout = java.awt.GridBagLayout()
		self.contentPane.setPreferredSize(Dimension(size[0], size[1]))
		self.contentPane.setSize(Dimension(size[0], size[1]))

		self.tabPane = None

	def addTabbedPane(self, tConstraints):
		# create and add the main tabbed pane:
		self.tabPane = ExtendedJTabbedPane()
		self.tabPane.addChangeListener(MyChangeListener(self.tabPane.tabValidityChecklist))
		self.tabPane.setPreferredSize(self.prefSize)
		self.tabPane.setSize(self.prefSize)
		self.contentPane.add(self.tabPane, tConstraints)

	# add new tabs to the main tabbed pain:
	def addTab(self, tabName, tabContents):
		self.tabMap[tabName] = tabContents
		self.tabPane.addTab(tabName, tabContents)
		tabIdx = self.tabPane.indexOfTab(tabName)
		self.tabPane.tabValidityChecklist.setdefault(tabIdx, [])

	def addTabValidityCheck(self, tabName, validityCheck):
		# check the validity of the data (normalization, deleted plates, etc.) before repainting tabs in the checklist
		tabIdx = self.tabPane.indexOfTab(tabName)
		self.tabPane.tabValidityChecklist.setdefault(tabIdx, [])
		self.tabPane.tabValidityChecklist[tabIdx].append(validityCheck)

	def selectTab(self, tabName):
		self.tabPane.setSelectedComponent(self.tabMap[tabName])
		self.tabPane.validate()

class ExtendedJTabbedPane(JTabbedPane):
	# just a JTabbedPane with a validity checklist added
	def __init__(self):
		JTabbedPane.__init__(self) #@UndefinedVariable
		self.tabValidityChecklist = {}

class MyChangeListener(ChangeListener):
	# just a ChangeListener that checks the validity when a change is detected
	def __init__(self, checkList):
		self.checkList = checkList

	def stateChanged(self, evt):
		# get the source that selected this
		source = evt.getSource()
		# get the index of the tab that was selected
		tabIdx = source.getSelectedIndex()
		# get the component that was selected
		tComp = source.getComponentAt(tabIdx)

		# load the list of items to be checked:
		vChecklist = source.tabValidityChecklist.get(tabIdx, [])
		if len(vChecklist) != 0:
			# apply the validity check for each item in the list:
			for vCheck in vChecklist:
				#print 'Applying ', vCheck
				apply(vCheck, [1])

class NewPanel(JPanel):
	# jython-accessible JPanel
	def __init__(self):
		JPanel.__init__(self) #@UndefinedVariable
		self.compDict = {}
		self.g = self.getGraphics()

	# add a component to the panel
	def addComponent(self, compName, comp, constraints=None):
		if constraints != None:
			self.add(comp, constraints)
		else:
			self.add(comp)

		self.compDict[compName] = comp

	# allow access to the panel component by name:
	def addComponentId(self, compName, comp):
		self.compDict[compName] = comp

	# get the component by name
	def getComp(self, compName):
		try:
			return(self.compDict[compName])
		except:
			return None

	# refresh/repaint the panel:
	def refresh(self):
		if self.g == None:
			self.g = self.getGraphics()
		self.update(self.g)


class ListBox(JScrollPane):
	# jython-accessible JScrollPane
	def __init__(self, header, size, selectionMode):
		# create the scrollPane
		JScrollPane.__init__(self) #@UndefinedVariable

		# create the list object
		self.myList = JList()
		self.myList.setSelectionMode(selectionMode)

		# make a header for the list and set the attributes
		self.header = JTextField(horizontalAlignment=swing.SwingConstants.LEFT)
		self.header.setText(header)
		self.header.setFont(Font.Helvetica12Bold)
		self.header.setEditable(0)
		self.setColumnHeaderView(self.header)

		# add the list to the scrollPane
		self.getViewport().setView(self.myList)

		# empty data Idx list:
		self.myDataList = []

		# allow storage of key-value pairs:
		self.kvp = {}

	# get/set methods for various features:
	def get_myList(self):
		return self.myList

	def get_myDataList(self):
		return self.myDataList

	def setData(self, data, dataId):
		self.myList.setListData(data)
		self.myDataList = dataId

	def setWidthByPrototype(self, string):
		#self.myList.setPreferredSize(Dimension(w,h))
		self.myList.setPrototypeCellValue(string)

	def setHolder(self, container):
		self.myContainer = container

	def getHolder(self):
		return(self.myContainer)

	def setField(self, key, value):
		self.kvp[key] = value

	def getField(self, key):
		return(self.kvp[key])

	def setHeader(self, string):
		hWidth = int(self.myList.getPreferredSize().getWidth())
		self.header = JTextField(preferredSize=(hWidth, 20), horizontalAlignment=swing.SwingConstants.LEFT)
		self.header.setText(string)
		self.header.setEditable(0)
		self.setColumnHeaderView(self.header)

	def addListSelectCallback(self, callback):
		listSelector = ExtendedListSelector(self, callback)
		self.myList.addMouseListener(listSelector)

# This class defines a JButton with an equal-width border on all 4 sides (default width = 1)
# The pane containing the button is returned by the class, and the JButton
# component can be accessed as <buttonPane>.button:
class BorderedButton(JPanel):
	def __init__(self, label, action, borderWidth=1, layout='box'):
		bw = borderWidth
		JPanel.__init__(self) #@UndefinedVariable
		if layout == 'box':
			self.setLayout((BoxLayout(self, BoxLayout.X_AXIS)))
		elif layout == 'gridBag':
			self.setLayout(GridBagLayout())
		else:
			print 'BorderedButton: Unknown button panel layout type: %s' % layout
			print ' (using JPanel default layout)'

		self.setBorder(BorderFactory.createEmptyBorder(bw, bw, bw, bw))
		self.button = ExtendableJButton(label, action)
		self.add(self.button)
		self.addKeyValuePair('bPane', self)

	def addKeyValuePair(self, key, value):
		self.button.kvp[key] = value

	def getValue(self, key):
		return self.button.getValue(key)

class ExtendableJButton(JButton):
	# JButton with a set of key/value pairs attached
	def __init__(self, label, action):
		JButton.__init__(self, label, actionPerformed=action) #@UndefinedVariable
		self.kvp = {}

	def setKeyValuePair(self, key, value):
		self.kvp[key] = value

	def getValue(self, key):
		try:
			return self.kvp[key]
		except:
			print 'Unkown key: %s' % key
			return None

class ExtendedListSelector(java.awt.event.MouseListener):
	# a MouseListener that allows for a callback to be executed when a list item is selected:
	def __init__(self, thisList, callback):
		self.list = thisList
		self.callback = callback

	def mouseClicked(self, evt):
		self.callback(self.list)

	# ignore other mouse events (these need to be defined for the mouse listener to work):
	def mouseEntered(self, evt):
		pass
	def mouseExited(self, evt):
		pass
	def mousePressed(self, evt):
		pass
	def mouseReleased(self, evt):
		pass
	def mouseDragged(self, evt):
		pass
	def mouseMoved(self, evt):
		pass

# Threshold sliders are JSliders that display the selected value in a small box near the slider
class ThresholdSlider(JPanel):
	def __init__(self, title, width, height, minThresh, maxThresh, initThresh,
				 orient=JSlider.VERTICAL, headerGap=None, textWidth=5, sliderSteps=N_THRESH_STEP):
		JPanel.__init__(self) #@UndefinedVariable
		self.threshTextField = None
		self.slider = None
		self.sliderSteps = sliderSteps    # number of slider steps
		self.minThresh = minThresh        # minimum slider value
		self.maxThresh = maxThresh        # maximum slider value
		self.stepSize = (maxThresh - minThresh) / sliderSteps     # the size of each slider step
		
		self.setSliderLayout(title, width, height, minThresh, maxThresh, initThresh, orient, headerGap, textWidth)

	def setSliderLayout(self, title, width, height, minThresh, maxThresh, initThresh, orient, headerGap, textWidth):
		# NOTE: this could probably be consolidataed into the _init_ method
		self.setLayout(BoxLayout(self, BoxLayout.Y_AXIS))
		self.setSize(Dimension(width, height))

		# put the label and textField in a Box:
		labelBox = Box(BoxLayout.Y_AXIS)

		# make the title:
		label = JLabel(title, JLabel.CENTER)
		label.setAlignmentX(0.5)

		# make the text formatter:
		threshFormat = new1point2Format()
		threshFormatter = NumberFormatter(threshFormat)
		threshFormatter.setMinimum(java.lang.Float(minThresh))
		threshFormatter.setMaximum(java.lang.Float(maxThresh))

		# figure out how to format the text
		threshTextField = JTextField(textWidth)
		if textWidth > 5:
			fmt = '%%%dd' % textWidth        # large -> integers
		else:
			fmt = '%%%d.2f' % textWidth      # small -> floats
		threshTextField.setText(fmt % initThresh)     # initialize the data being displayed
		threshTextField.setHorizontalAlignment(JTextField.CENTER)
		threshTextField.setMaximumSize(Dimension(textWidth * 10, 20))

		self.threshTextField = threshTextField

		# label the slider
		labelBox.add(label)
		if headerGap != None:
			labelBox.add(Box.createVerticalStrut(headerGap))
		labelBox.add(threshTextField)

		# make the slider:
		sInit = int(((initThresh - minThresh) / (maxThresh - minThresh)) * self.sliderSteps)
		#print 'min, init, max:', minThresh, initThresh, maxThresh
		slider = JSlider(orient, 0, self.sliderSteps, sInit)  # create the slider with the correct orientation
		slider.addChangeListener(ThreshSliderListener(title, threshTextField, minThresh, maxThresh, self.sliderSteps))
		slider.setAlignmentX(0.5)
		self.slider = slider

		# package them up into a JPanel:
		self.add(labelBox)
		#self.add(self.threshTextField)
		self.add(self.slider)

	def addSliderSideEffect(self, effectFunc, params=()):
		# causes a 'side effect' to occur when a slider is moved (redisplay a graph, select items from a list, etc.)
		listeners = self.slider.getChangeListeners()
		for listener in listeners:
			listener.addSliderSideEffect(effectFunc, params)

	def updateLimits(self, minThresh, maxThresh, initThresh=None):
		# readjust the limits of the slider
		if initThresh == None:
			initThresh = (minThresh + maxThresh) / 2.0

		listeners = self.slider.getChangeListeners()
		for listener in listeners:
			listener.updateLimits(minThresh, maxThresh, initThresh)
			listener.stateChanged(ChangeEvent(self))

	def updateSliderValue(self, fract):
		nSteps = int(self.sliderSteps * fract)
		self.slider.setValue(nSteps)

	def getValue(self):
		# return the value selected by the slider:
		return self.slider.getValue()

	def getDisplayedValue(self):
		#print 'getDisplayedValue'
		#print '	  value:',self.slider.getValue()
		#print '  low limit:',self.minThresh
		#print ' high limit:',self.maxThresh
		#print '	   step:',self.stepSize
		return self.minThresh + (self.slider.getValue()*self.stepSize)

	def getValueIsAdjusting(self):
		# NOTE: not sure why this doesn't just call getValueIsAdjusting
		return 0
		#return self.slider.getValueIsAdjusting()

class ThresholdSliderWithLimits(ThresholdSlider):
	# An extended Threshold slider with the min and max displayed at the ends of the slider, used on the Data Analysis tab
	def __init__(self, title, width, height, minThresh, maxThresh, initThresh,
				 orient=JSlider.VERTICAL, side=JTextField.RIGHT, sliderSteps=N_THRESH_STEP):
		JPanel.__init__(self) #@UndefinedVariable
		self.threshTextField = None
		self.minTextField = None
		self.maxTextField = None
		self.slider = None
		self.sliderSteps = sliderSteps
		self.minThresh = minThresh
		self.maxThresh = maxThresh
		self.stepSize = (maxThresh - minThresh) / sliderSteps
		if orient == JSlider.VERTICAL:
			self.setSliderLayout(title, width, height, minThresh, maxThresh, initThresh, side)
		else:
			self.setSliderLayoutHoriz(title, width, height, minThresh, maxThresh, initThresh)

	def getThresh(self):
		sVal = self.slider.getValue()
		thresh = self.minThresh + self.stepSize * sVal
		return thresh

	def setSliderLayout(self, title, width, height, minThresh, maxThresh, initThresh, side):
		self.setLayout(GridBagLayout())
		self.setSize(Dimension(width, height))

		# put the label and textField in a Box:
		labelBox = Box(BoxLayout.Y_AXIS)

		# make the title:
		label = JLabel(title, JLabel.CENTER)
		label.setAlignmentX(0.5)

		# make the text formatter:
		threshFormat = new3point2Format()
		threshFormatter = NumberFormatter(threshFormat)
		threshFormatter.setMinimum(java.lang.Float(minThresh))
		threshFormatter.setMaximum(java.lang.Float(maxThresh))

		# make the text box:
		threshTextField = JTextField(5)
		threshTextField.setText('%5.2f' % initThresh)
		threshTextField.setHorizontalAlignment(JTextField.CENTER)

		self.threshTextField = threshTextField

		labelBox.add(label)
		labelBox.add(Box.createVerticalStrut(5))
		labelBox.add(threshTextField)
		labelBox.add(Box.createVerticalStrut(5))

		# make min/max text fields:
		minTextField = JTextField(8)
		minTextField.setBackground(self.getBackground())
		minTextField.setBorder(BorderFactory.createEmptyBorder())
		#minTextField.setValue(minThresh)
		minTextField.setText('%5.2f' % minThresh)
		minTextField.setColumns(8)
		minTextField.setMaximumSize(Dimension(60, 20))
		self.minTextField = minTextField

		maxTextField = JTextField(8)
		maxTextField.setBackground(self.getBackground())
		maxTextField.setBorder(BorderFactory.createEmptyBorder())
		maxTextField.setText('%5.2f' % maxThresh)
		maxTextField.setColumns(6)
		maxTextField.setMaximumSize(Dimension(60, 20))
		self.maxTextField = maxTextField

		# make the slider:
		sInit = int(((initThresh - minThresh) / (maxThresh - minThresh)) * self.sliderSteps)
		#print 'min, init, max:', minThresh, initThresh, maxThresh
		slider = JSlider(JSlider.VERTICAL)
		slider.addChangeListener(ThreshSliderListener(title, threshTextField, minThresh, maxThresh))
		slider.setAlignmentX(0.5)
		self.slider = slider

		# package them up into the JPanel:
		threshInsets = Insets(0, 0, 0, 0)
		if side == JTextField.RIGHT:
			minTextField.setHorizontalAlignment(JTextField.LEFT)
			maxTextField.setHorizontalAlignment(JTextField.LEFT)
			constraints = DefaultGridbagConstraints()

			# Threshold setting
			constraints.insets = threshInsets
			constraints.anchor = GridBagConstraints.NORTH
			constraints.gridwidth = GridBagConstraints.REMAINDER
			constraints.fill = GridBagConstraints.HORIZONTAL
			self.add(labelBox, constraints)

			# Slider
			constraints.insets = Insets(0, 0, 0, 5)
			constraints.gridwidth = 1
			constraints.gridy = 1
			constraints.fill = GridBagConstraints.NONE
			constraints.anchor = GridBagConstraints.NORTHEAST
			self.add(self.slider, constraints)

			# Lower limit
			constraints.insets = Insets(0, 0, 0, 0)
			constraints.gridx = 1
			constraints.anchor = GridBagConstraints.SOUTH
			constraints.fill = GridBagConstraints.HORIZONTAL
			self.add(self.minTextField, constraints)

			# Upper limit
			constraints.anchor = GridBagConstraints.NORTH
			self.add(self.maxTextField, constraints)
		else:
			minTextField.setHorizontalAlignment(JTextField.RIGHT)
			maxTextField.setHorizontalAlignment(JTextField.RIGHT)

			constraints = DefaultGridbagConstraints()

			# Threshold setting
			constraints.insets = threshInsets
			constraints.fill = GridBagConstraints.HORIZONTAL
			constraints.gridwidth = GridBagConstraints.REMAINDER
			constraints.anchor = GridBagConstraints.NORTH
			constraints.insets = Insets(0, 5, 0, 0)
			self.add(labelBox, constraints)

			# Slider
			constraints.insets = Insets(0, 0, 0, 0)
			constraints.fill = GridBagConstraints.NONE
			constraints.gridx = 1
			constraints.gridwidth = 1
			constraints.anchor = GridBagConstraints.NORTHWEST
			constraints.gridy = 1
			self.add(self.slider, constraints)

			# Lower limit
			constraints.insets = Insets(0, 0, 0, 0)
			constraints.gridx = 0
			constraints.anchor = GridBagConstraints.SOUTH
			constraints.fill = GridBagConstraints.HORIZONTAL
			self.add(self.minTextField, constraints)

			# Upper limit
			constraints.anchor = GridBagConstraints.NORTH
			self.add(self.maxTextField, constraints)

	def setSliderLayoutHoriz(self, title, width, height, minThresh, maxThresh, initThresh):
		# horizontal slider layout		
		self.setLayout(GridBagLayout())
		self.setSize(Dimension(width, height))
		# put the label and textField in a Box:
		labelBox = Box(BoxLayout.Y_AXIS)

		# make the title:
		label = JLabel(title, JLabel.CENTER)
		label.setAlignmentX(0.5)

		# make the text formatter:
		threshFormat = new3point2Format()
		threshFormatter = NumberFormatter(threshFormat)
		threshFormatter.setMinimum(java.lang.Float(minThresh))
		threshFormatter.setMaximum(java.lang.Float(maxThresh))

		threshTextField = JTextField(5)
		threshTextField.setText('%5.2f' % initThresh)
		threshTextField.setHorizontalAlignment(JTextField.CENTER)

		self.threshTextField = threshTextField

		labelBox.add(label)
		labelBox.add(Box.createVerticalStrut(5))
		labelBox.add(threshTextField)
		labelBox.add(Box.createVerticalStrut(5))

		# make min/max text fields:
		minTextField = JTextField(6)
		minTextField.setBackground(self.getBackground())
		minTextField.setBorder(BorderFactory.createEmptyBorder())
		minTextField.setText('%5.2f' % minThresh)
		self.minTextField = minTextField

		#maxTextField = JFormattedTextField(threshFormatter)
		maxTextField = JTextField(6)
		maxTextField.setBackground(self.getBackground())
		maxTextField.setBorder(BorderFactory.createEmptyBorder())
		maxTextField.setText('%5.2f' % maxThresh)
		self.maxTextField = maxTextField

		# make the slider:
		sInit = int(((initThresh - minThresh) / (maxThresh - minThresh)) * self.sliderSteps)
		#print 'min, init, max:', minThresh, initThresh, maxThresh
		slider = JSlider(JSlider.HORIZONTAL)
		slider.addChangeListener(ThreshSliderListener(title, threshTextField, minThresh, maxThresh))
		slider.setAlignmentX(0.5)
		self.slider = slider

		# package them up into the JPanel:
		threshInsets = Insets(0, 0, 0, 0)
		minTextField.setHorizontalAlignment(JTextField.LEFT)
		maxTextField.setHorizontalAlignment(JTextField.RIGHT)
		constraints = DefaultGridbagConstraints()

		# Threshold setting
		constraints.insets = threshInsets
		constraints.anchor = GridBagConstraints.NORTH
		constraints.gridwidth = GridBagConstraints.REMAINDER
		constraints.fill = GridBagConstraints.NONE
		self.add(labelBox, constraints)

		# Slider
		constraints.insets = Insets(0, 5, 0, 5)
		constraints.gridwidth = 5
		constraints.gridx = 0
		constraints.gridy = 1
		constraints.fill = GridBagConstraints.HORIZONTAL
		constraints.anchor = GridBagConstraints.NORTHEAST
		self.add(self.slider, constraints)

		# Lower limit
		constraints.insets = Insets(0, 3, 0, 0)
		constraints.gridwidth = 1
		constraints.gridx = 0
		constraints.gridy = 2
		constraints.anchor = GridBagConstraints.NORTHWEST
		constraints.fill = GridBagConstraints.NONE
		self.add(self.minTextField, constraints)

		# Upper limit
		constraints.insets = Insets(0, 0, 0, 3)
		constraints.anchor = GridBagConstraints.NORTHEAST
		constraints.gridx = 4
		self.add(self.maxTextField, constraints)

	def updateLimits(self, minThresh, maxThresh, initThresh=None):
		# re-scale the slider with new min and max limits
		if initThresh == None:
			initThresh = (minThresh + maxThresh) / 2.0

		#self.minTextField.setValue(minThresh)
		#self.maxTextField.setValue(maxThresh)
		self.minTextField.setText('%5.2f' % minThresh)
		self.maxTextField.setText('%5.2f' % maxThresh)
		self.minThresh = minThresh
		self.maxThresh = maxThresh
		self.stepSize = (maxThresh - minThresh) / self.sliderSteps

		listeners = self.slider.getChangeListeners()
		for listener in listeners:
			listener.updateLimits(minThresh, maxThresh, initThresh)
			listener.stateChanged(ChangeEvent(self))

class ThreshSliderListener(ChangeListener):
	# a ChangeListener called when th slider position is moved	
	def __init__(self, name, threshTextField, threshMin, threshMax, nSteps=N_THRESH_STEP):
		self.threshTextField = threshTextField
		self.threshMin = threshMin
		self.threshMax = threshMax
		self.nSteps = nSteps
		self.step = (threshMax - threshMin) / nSteps
		self.effect = []
		self.name = name

	def updateLimits(self, threshMin, threshMax, initThresh):
		# update the slider limits		
		self.threshMin = threshMin
		self.threshMax = threshMax
		self.step = (threshMax - threshMin) / self.nSteps

	def addSliderSideEffect(self, effectFunc, params):
		# add a 'side effect that occurs when the slider is moved
		self.effect.append([effectFunc, params])

	def stateChanged(self, evt):
		# action performed when the state of the slider is changed
		source = evt.getSource()
		sVal = source.getValue()
		thresh = self.threshMin + self.step * sVal
		self.threshTextField.setText('%6.2f' % thresh)
		masterObject.setField(self.name, thresh)

		# When the value is adjusting, just update the display, ignore all other side effects:
		if source.getValueIsAdjusting():
			return

		if len(self.effect):
			for effect in self.effect:
				effectFunc = effect[0]
				if len(effect[1]):	 # if there is at least one argument
					apply(effectFunc, effect[1][0:])
				else:
					effectFunc(thresh)

# create a number formater to produce numbers of the form x.xx	
def new1point2Format():
	format = NumberFormat.getNumberInstance()
	format.setMaximumFractionDigits(2)
	format.setMinimumFractionDigits(2)
	format.setMaximumIntegerDigits(1)
	format.setMinimumIntegerDigits(1)
	return format

# create a number formater to produce numbers of the form xxx.xx	
def new3point2Format():
	format = NumberFormat.getNumberInstance()
	format.setMaximumFractionDigits(2)
	format.setMinimumFractionDigits(2)
	format.setMaximumIntegerDigits(3)
	format.setMinimumIntegerDigits(1)
	return format

# create a number formater to produce numbers of the form x.xxx	
def new1point3Format():
	format = NumberFormat.getNumberInstance()
	format.setMaximumFractionDigits(3)
	format.setMinimumFractionDigits(3)

	format.setMaximumIntegerDigits(1)
	format.setMinimumIntegerDigits(1)
	return format

# create a number formater to produce numbers of the form xx.xx	
def new2point2Format():
	# ...which is the same as a percent formatter
	format = NumberFormat.getPercentInstance()
	return format

class NonEditableTableModel(DefaultTableModel):
	# a TableMode with the ability to make the entries in the table non-editable
	def __init__(self, editable=1):
		DefaultTableModel.__init__(self) #@UndefinedVariable
		self.isEditable = editable
		self.key2row = {}
		self.row2key = {}
		self.keyColor = {}

	def addFlaggedKey(self, key, rowIdx, color):
		self.row2key[rowIdx] = key
		self.key2row[key] = rowIdx
		self.keyColor[key] = color

	def removeFlaggedKey(self, key):
		rowIdx = self.key2row.get(key, None)
		if rowIdx != None:
			del self.row2key[rowIdx]
			del self.key2row[key]
			del self.keyColor[key]

	def clearFlaggedKeys(self):
		self.key2row = {}
		self.row2key = {}
		self.keyColor = {}

	def moveFlaggedKey(self, key, rowIdx):
		oldRowIdx = self.key2row.get(key, None)
		if oldRowIdx != None:
			del self.row2key[oldRowIdx]
			self.row2key[rowIdx] = key
			self.key2row[key] = rowIdx

	def isCellEditable(self, row, col):
		return self.isEditable

class TableCellRendererWithRowColor(DefaultTableCellRenderer):
	# a table cell renderer with the ability to set the color of the cells
	def __init__(self):
		DefaultTableCellRenderer.__init__(self) #@UndefinedVariable

	def getTableCellRendererComponent(self, table, value, isSelected, hasFocus, row, col):
		# if there is an entry in model.row2key for this row, use the 'removed' color for the 
		#  cell background, otherwise use WHITE:
		model = table.getModel()
		bgColor = model.keyColor.get(model.row2key.get(row, None), Color.WHITE)
		self.setBackground(bgColor)
		comp = DefaultTableCellRenderer.getTableCellRendererComponent(self, table, value,
																	  isSelected, hasFocus, row, col)
		# to distinguish 'removed' rows from others, make the font italic:
		oldFontStyle = comp.getFont().getStyle()
		if bgColor == Color.WHITE:
			comp.setFont(Font.Ariel12Plain)
			if table.isRowSelected(row):
				# light blue indicates that the row is selected
				table.setSelectionBackground(Color.newColor(184, 207, 229))
		else:
			comp.setFont(Font.Ariel12Italic)
			if table.isRowSelected(row):
				# light purple (light blue + light red) indicates that the row is selected, but disabled
				table.setSelectionBackground(Color.newColor(225, 181, 249))

		return comp

class NormTablePane(JScrollPane):
	# multi-column table used on several of the tabs
	def __init__(self, editable=1):
		self.tableModel = NonEditableTableModel(editable)
		# set up the table and the table attributes
		self.table = JTable(self.tableModel)
		self.table.setColumnModel(DefaultTableColumnModel())
		self.tableModel = self.table.getModel()
		self.table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION)
		JScrollPane.__init__(self, self.table) #@UndefinedVariable
		self.colNameList = []
		self.tRend = TableCellRendererWithRowColor()
		self.table.setDefaultRenderer(Object().getClass(), self.tRend)
		self.rowKey = {}
		self.headerDefaultColor = self.table.getTableHeader().getBackground()
		self.headerDefaultFont = self.table.getTableHeader().getFont()
		self.selectionBgDefault = self.table.getSelectionBackground()

	def setColumnNames(self, colNameList):
		# label the columns
		cIdx = 0
		for colName in colNameList:
			self.tableModel.addColumn(colName)
			self.colNameList.append(colName)
			cIdx += 1

	def setPreferredColumnWidths(self, widthList):
		# set the preferred column starting widths
		self.table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF)
		for colIdx in range(len(widthList)):
			col = self.table.getColumnModel().getColumn(colIdx)
			col.setPreferredWidth(widthList[colIdx])

	def setColumnHorizontalAlignment(self, alignment):
		# set the horizontal alignment of the columns
		for colIdx in range(len(self.colNameList)):
			col = self.table.getColumnModel().getColumn(colIdx)
			cRend = col.getCellRenderer()
			if cRend != None:
				cRend.setHorizontalAlignment(alignment)
			else:
				#cRend = DefaultTableCellRenderer()
				cRend = TableCellRendererWithRowColor()
				cRend.setHorizontalAlignment(alignment)
				col.setCellRenderer(cRend)

	def setColumnBackground(self, colIdx, bgColor):
		# set the default background color of each column 
		col = self.table.getColumnModel().getColumn(colIdx)
		cRend = col.getCellRenderer()
		if cRend != None:
			cRend.setBackground(bgColor)
		else:
			#cRend = DefaultTableCellRenderer()
			cRend = TableCellRendererWithRowColor()
			cRend.setBackground(bgColor)
			col.setCellRenderer(cRend)

	def setTableSelectionBackground(self, bgColor=None):
		# set the table background color to the provided color, or the default
		if bgColor == None:
			self.table.setSelectionBackground(self.selectionBgDefault)
		else:
			self.table.setSelectionBackground(bgColor)
		self.table.update(self.table.getGraphics())

	def setHeaderBackground(self, headerColor=None):
		# set the background color o fthe header
		tHead = self.table.getTableHeader()
		if headerColor == None:
			tHead.setBackground(self.headerDefaultColor)
		else:
			tHead.setBackground(headerColor)
		tHead.update(tHead.getGraphics())

	def setHeaderFont(self, headerFont=None):
		# set the font of the header
		tHead = self.table.getTableHeader()
		if headerFont == None:
			tHead.setFont(self.headerDefaultFont)
		else:
			tHead.setFont(headerFont)
		tHead.update(tHead.getGraphics())

	def addRow(self, rowData, rowId=None):
		# add a row to the table
		self.tableModel.addRow(rowData)
		rowIdx = self.tableModel.getRowCount() - 1
		if rowId == None:
			self.rowKey[rowIdx] = rowIdx
			rowId = rowIdx
		else:
			self.rowKey[rowIdx] = rowId
		return rowIdx

	def getRow(self, rowIdx):
		# get the data in the columns of the selected row
		rowOut = []
		for colIdx in range(len(self.colNameList)):
			rowOut.append(self.tableModel.getValueAt(rowIdx, colIdx))
		return rowOut

	def getRowKey(self, rowIdx):
		# get the 'key' reference for the selected row
		if rowIdx == -1:
			return - 1
		elif len(self.rowKey) == 0:
			return - 1
		else:
			try:
				return self.rowKey[rowIdx]
			except:
				return self.rowKey[0]


	def getKeyRow(self, key):
		# get the row index of the row referenced by the key provided
		rowList = self.rowKey.keys()
		rowIdx = -1
		for row in rowList:
			if self.rowKey[row] == key:
				rowIdx = row
				break
		return rowIdx

	def clearList(self):
		# clear the list of selected rows
		nRows = self.getRowCount()
		for i in range(nRows):
			self.removeRow(0)

		self.tableModel.clearFlaggedKeys()
		self.rowKey = {}

	def removeRow(self, rowIdx):
		# remove a row from the table
		self.tableModel.removeRow(rowIdx)

	def getRowCount(self):
		# count the rows in the table
		return self.tableModel.getRowCount()

	def setSelectionMode(self, selMode):
		# set the selection mode of the table (multiple row, single row, etc.)
		self.table.tableModel.setSelectionMode(selMode)

	def addListSelectionListener(self, listener):
		# add a selection listener to the table
		self.table.getSelectionModel().addListSelectionListener(listener)

	def addHeaderListener(self, listener):
		# add a listener to the header (for sorting columns, etc)
		self.table.getTableHeader().addMouseListener(listener)

	def addTableKeyboardListener(self, listener):
		# add a keyboard listener to the table (for deleting rows using the DELETE key, etc.)
		self.table.addKeyListener(listener)

	def sortTableContents(self, cIdx= -1, default='Virus Plate'):
		# get the index of the main sort column:
		if cIdx == -1:
			try:
				cIdx = self.colNameList.index(default)
			except:
				print 'No sort column named "%s"' % default
				return

		# get the index of the default (fallback) sort column:
		try:
			dIdx = self.colNameList.index(default)
		except:
			print 'No default column named "%s"' % default
			return

		model = self.table.getModel()
		nRow = model.getRowCount()
		tableData = model.getDataVector()
		# create the list of items to be sorted
		sortList = []
		for i in range(nRow):
			sortList.append([tableData[i][cIdx], tableData[i][dIdx], i])

		# sort the list:
		sortList.sort(fieldSort)

		# build a new (sorted) list of rows:
		sortedTableData = []
		sortedRowKeys = {}
		for i in range(nRow):
			# row swap
			idx = sortList[i][2]

			# a new row must be constructed from the elements of the tableData row:
			newRow = []
			for col in range(len(self.colNameList)):
				newRow.append(tableData[idx][col])
			sortedTableData.append(newRow)

			sortedRowKeys[i] = self.rowKey[idx]
			self.tableModel.moveFlaggedKey(self.rowKey[idx], i)

		self.rowKey = sortedRowKeys

		# using setValueAt(), rather than setDataVector(), keeps columns from
		# getting reformatted:
		for row in range(nRow):
			for col in range(len(self.colNameList)):
				model.setValueAt(sortedTableData[row][col], row, col)

		self.repaint()

def fieldSort(a, b):
	if a[0] < b[0]:
		return - 1
	elif a[0] > b[0]:
		return 1
	elif a[1] < b[1]:
		return - 1
	elif a[1] > b[1]:
		return 1
	else:
		return 0

# This class allows a set of radio buttons to be grouped, labeled, and have actions assigned to them
class RadioButtonUtils:
	def __init__(self):
		self.panel = None
		self.default = None
		self.panelBgColor = Color.LIGHT_GRAY
		self.buttonBgColor = Color.LIGHT_GRAY
		self.title = None
		self.buttonMap = {}

	def createRadioButtonGrouping(self, buttonList, title=None, actionListener=None, defaultSelect=0, nCol=1):
		# create a grouping of radio buttons
		self.panel = JPanel(GridLayout(0, nCol))
		if title:
			border = BorderFactory.createTitledBorder(title)
			self.panel.setBorder(border)
			self.title = title

		# make a button group, using the swing ButtonGroup 
		self.group = ButtonGroup()
		self.default = buttonList[defaultSelect]
		bNum = 0
		for buttonName in buttonList:
			# add each button in buttonList to the group
			if bNum == defaultSelect:
				button = JRadioButton(buttonName, 1)    # default "selected" button
			else:
				button = JRadioButton(buttonName, 0)    # "unselected" buttons

			self.panel.add(button)
			self.group.add(button)
			if actionListener:
				# add the actionListener to each button (if one is provided)
				button.addActionListener(actionListener)

			# create a mapping of button names to button objects
			self.buttonMap[buttonName] = button

			bNum += 1

		return self.panel

	def getDefault(self):
		# the default button
		return self.default

	def setBgColor(self, color):
		# background color for the button group
		self.panelBgColor = color
		if self.panel != None:
			self.panel.setBackground(color)

	def setButtonBgColor(self, color):
		# background color for the button
		self.buttonBgColor = color
		for button in self.buttonMap.keys():
			self.buttonMap[button].setBackground(color)

	def clickButton(self, buttonName):
		# the "click" action for the button
		try:
			self.buttonMap[buttonName].doClick()
		except:
			print 'Invalid buttonName: %s' % buttonName

	def setSelectedButton(self, buttonName):
		# set the state of the button to "selected"
		try:
			self.buttonMap[buttonName].setSelected(1)
		except:
			print 'Invalid buttonName: %s' % buttonName

	def clickDefaultButton(self):
		# select the default button
		self.buttonMap[self.default].doClick()

class CustomActionListener(java.awt.event.ActionListener):
	# an extended ActionListener tha allows for a list of events to be 
	# processed, rather than a single event
	
	def __init__(self, actionFunc, args=[], buttonGroup=None):
		# if a buttonGroup is given, the default button is used to initialize lastAction,
		# so nothing happens if the default button is clicked first.
		self.actionFunc = actionFunc
		self.args = args

	def actionPerformed(self, evt):
		# perform the action assigned to this event
		apply(self.actionFunc, [evt])

	def fireListener(self, actionCommand=None):
		# cause the actionListener to fire
		actionString = '%s' % actionCommand
		evt = java.awt.event.ActionEvent(self, 0, actionString)
		apply(self.actionFunc, [evt])

class CustomChangeListener(swing.event.ChangeListener):
	# an extended ChangeListener tha allows for a list of events to be 
	# processed, rather than a single event
	def __init__(self, actionFunc, args=[]):
		self.actionFunc = actionFunc
		self.args = args

	def stateChanged(self, evt):
		apply(self.actionFunc, [evt])

class CustomTableModelListener(TableModelListener):
	# an extended TableModelListener tha allows for a list of events to be 
	# processed, rather than a single event
	def __init__(self, actionFunc, args=[]):
		self.actionFunc = actionFunc
		self.args = args

	def tableChanged(self, evt):
		apply(self.actionFunc, [evt])

class CustomListSelectionListener(swing.event.ListSelectionListener):
	# an extended ListSelectionListener tha allows for a list of events to be 
	# processed, rather than a single event
	def __init__(self, actionFunc, args=[]):
		self.actionFunc = actionFunc
		self.args = args

	def valueChanged(self, evt):
		apply(self.actionFunc, [evt])

class TableKeyboardListener(java.awt.event.KeyListener):
	# if a buttonGroup is given, the default button is used to initialize lastAction,
	# so nothing happens if the default button is clicked first.

	def __init__(self, table, actionFunc, args=[]):
		self.actionFunc = actionFunc
		self.args = args
		self.table = table

	def keyTyped(self, evt):
		# do nothing when a key is typed
		pass

	def keyPressed(self, evt):
		# do nothing when a key is pressed
		pass

	def keyReleased(self, evt):
		# perform the assigned action function(s) when the key is released
		apply(self.actionFunc, [evt, self.table])

class TableListSelectionListener(swing.event.ListSelectionListener):
	# an extended ListSelectionListener tha allows for a list of events to be 
	# processed, rather than a single event, as well as associating the action
	# with a specific table
	def __init__(self, table, actionFunc, args=[]):
		self.actionFunc = actionFunc
		self.args = args
		self.table = table

	def valueChanged(self, evt):
		#print 'TableListSelectionListener:evt', evt
		apply(self.actionFunc, [evt, self.table])

class HeaderMouseListener(MouseListener):
	# a mouse listener to perform an action when a table header is clicked
	def __init__(self, table, actionFunc, args=[]):
		self.actionFunc = actionFunc
		self.args = args
		self.table = table

	def mouseClicked(self, evt):
		# perform the assigned action only when the mouse button is clicked
		apply(self.actionFunc, [evt, self.table])

	def mouseEntered(self, evt):
		# do nothing on a mouse entering a region
		pass

	def mouseExited(self, evt):
		# do nothing on a mouse exiting a region
		pass

	def mousePressed(self, evt):
		# do nothing on a mouse button is pressed
		pass

	def mouseReleased(self, evt):
		# do nothing on a mouse button is released
		pass

class OutputFileFilter(FileFilter):
	# a customized FileFilter with an internal list of valid file extensions
	def __init__(self):
		self.extList = []

	def accept(self, f):
		# allow directories:
		if f.isDirectory():
			return 1

		# get the file extension:
		fExt = os.path.splitext(f.getAbsolutePath())[1]

		# scan the list of allowable extensions. If the extension matches one of them,
		# return TRUE (1), otherwise return FALSE (0):
		for ext in self.extList:
			if fExt == ext:
				return 1

		return 0

	def addExtension(self, ext):
		# add an extension to the list of allowable extensions
		if self.extList.count(ext) == 0:
			self.extList.append(ext)

	def getDescription(self):
		# get the list of allowable extensions
		outstr = ''
		for ext in self.extList:
			if outstr == '':
				outstr = ext
			else:
				outstr = '%s, %s' % (outstr, ext)

		return ('%s files only' % outstr)

#####################
#  HEAT MAP PANEL   #
#####################
###
#	ColorScale methods:
#	each takes a single value [0.0,1.0]
#	and returns an awt.Color
###
def _redGreen(value):
	# red/green color scale generally used for heat maps
	if value == -1:
		return COLOR_UNDEFINED
	else:
		return Color.newColor(1 - value, value, 0)

def _fullColor(value):
	# a full-color range (through the spectrum)
	if value == -1:
		return COLOR_UNDEFINED

	r = 1.5 - abs(1 - 4 * value)
	g = 1.5 - abs(2 - 4 * value)
	b = 1.5 - abs(3 - 4 * value)

	if r < 0:
		r = 0
	elif r > 1:
		r = 1

	if g < 0:
		g = 0
	elif g > 1:
		g = 1

	if b < 0:
		b = 0
	elif b > 1:
		b = 1

	return java.awt.Color(r, g, b)

def _blueYellow(value):
	# a blue-yellow color scale (for the color-blind)
	rg = math.sin(value * math.pi / 2)
	b = math.cos(value * math.pi / 2)

	return java.awt.Color(rg, rg, b)

ColorScales = { 'Red-green': _redGreen, 'Full-spectrum': _fullColor, 'Blue-yellow': _blueYellow }

#####################
#  HEAT MAP PANEL   #
#####################
class HeatPanel(JPanel, ChangeListener):
	# a 'heat-map' plate layout - allows for a plate layout to be shown, 
	# with each well painted in a specific color and shape within a square representing the well.
	def __init__(self, nRow, nCol, title, wellSize=15, colorScale=_redGreen):
		JPanel.__init__(self, GridLayout(nRow, nCol)) #@UndefinedVariable
		lineBorder = BorderFactory.createLineBorder(Color.BLACK)
		self.lineBorder = lineBorder
		self.setBorder(BorderFactory.createTitledBorder(lineBorder, title, TitledBorder.LEFT,
														TitledBorder.ABOVE_TOP))
		self.wellMap = {}
		self.minThresh = 0.0
		self.maxThresh = 1.0
		self.plateData = None
		self.plate = None
		self.trimPctile = 0
		# well names will be initialized with rows labeled
		# alphabetically, and columns labeled with 2-digit
		# numbers (e.g., A01, C12, etc.)
		wellDim = Dimension(wellSize, wellSize)

		# build the control well polygon shape:
		if wellSize == 15:
			# a nice octagon
			poly = Polygon()
			poly.addPoint(0, 4)
			poly.addPoint(0, 9)
			poly.addPoint(5, 14)
			poly.addPoint(9, 14)
			poly.addPoint(14, 9)
			poly.addPoint(14, 4)
			poly.addPoint(10, 0)
			poly.addPoint(4, 0)
		else:
			# a basic circle:
			cDiam = wellSize - 2.0
			poly = EllipseDouble(0, 0, cDiam, cDiam)

		#The method to be used for picking colors.
		self.colorScale = colorScale

		# set the name, size, etc of each well in the plate map
		for row in range(nRow):
			for col in range(nCol):
				rowStr = '%s' % ALPHABET[row]
				colStr = '%02d' % (col + 1)
				#well = JPanel()
				well = JWellPanel(poly)
				well.setPreferredSize(wellDim)
				well.setMinimumSize(wellDim)
				well.setSize(wellDim)
				lw = 0
				tw = 0
				if row == 0:
					tw = 1
				if col == 0:
					lw = 1
				well.setBorder(BorderFactory.createMatteBorder(tw, lw, 1, 1, Color.WHITE))
				self.wellMap.setdefault(rowStr, {})
				self.wellMap[rowStr][colStr] = well
				self.add(well)

	def setTrimPctile(self, trimPctile):
		self.trimPctile = trimPctile

	def paintWells(self, plateData=None, plate=None, normColor=0):
		# plateData is a map with data organized as [row][col]
		# normColor is a flag indicating whether to normalize the plateData
		#	to its min/max values
		# trimPctile indicates the percentile of values to consider 'off-scale' from
		#	the top and bottom of the the color scale

		if plateData != None:
			# initialize the plate data to the input "plateData" parameter
			self.plateData = plateData
		else:
			# ... or get the stored plate data
			plateData = self.plateData

		if plate != None:
			# initialise the plate to the input "plate" parameter
			self.plate = plate
		else:
			plate = self.plate

		# if a plate is given, get the associated virusPlate:
		if plate:
			try:
				vPlate = plate.virusPlate
				#print 'plate %s, virusPlate: ' % plate.plateName, vPlate
			except:
				print 'Unable to load virusPlate for plate %s' % plate.plateName
				vPlate = None
				return
		else:
			vPlate = None

		# normalize the data for display, if necessary:
		if normColor:
			# scan the data for its min/max values:
			minVal = None
			maxVal = None
			valList = []
			for row in plateData.keys():
				for col in plateData[row].keys():
					val = plateData[row][col]
					if val != None:
						valList.append(val)
						if maxVal == None or val > maxVal:
							maxVal = val
						if minVal == None or val < minVal:
							minVal = val

			# create storage for the normalized data and fill it:
			normData = {}

			# compute the max-min difference
			if maxVal == None or minVal == None or minVal == maxVal:
				minMaxDiff = None
			elif self.trimPctile == 0:
				minMaxDiff = maxVal - minVal
			else:
				# compute limits based on (n)th/(100-n)th percentile values:
				valList.sort()
				nItems = len(valList)
				pctLo = int((nItems / 100.0) * self.trimPctile)
				pctHi = int((nItems / 100.0) * (100 - self.trimPctile))
				minVal = valList[pctLo]
				maxVal = valList[pctHi]
				minMaxDiff = maxVal - minVal
				#print 'Clipped: low: %d, high: %d' % (pctLo, nItems-pctHi)

			# get the normalized 'color' value for each well (a value between 0 and 1)
			for row in plateData.keys():
				for col in plateData[row].keys():
					normData.setdefault(row, {})
					if plateData[row][col] == None or minMaxDiff == None:
						normData[row][col] = -1
					else:
						normData[row][col] = (plateData[row][col] - minVal) / float(minMaxDiff)
		else:
			minVal = 0.0
			maxVal = 1.0
			normData = plateData

		# set the background color of each well:
		# Note: this will always create a 384-well plate outline. If the displayed 
		#       plate has fewer wells, the "missing" wells will be filled with light gray.
		for row in ROWLIST:
			for col in COLLIST:
				well = self.wellMap[row][col]

				try:
					val = normData[row][col]
					# set the well color based on the threshold sliders:
					if val > self.maxThresh:
						well.setFgColor(HIGH_COLOR)
					elif val < self.minThresh:
						well.setFgColor(LOW_COLOR)
					else:
						well.setFgColor(self.colorScale(val))
				except:
					#well.setFgColor(Color.LIGHT_GRAY)
					well.setFgColor(java.awt.Color(230, 230, 250))
					well.setType(1)
					well.setToolTipText('')
					continue

				# set the well type based on the virusPlate well type:
				#vPlate.well[row][col].printWell(row, col)
				if vPlate != None and vPlate.well[row][col].type == 0:
					well.setType(0)	   # EMPTY wells
				elif (vPlate != None and vPlate.well[row][col].type == 1) or plate == None:
					well.setType(1)    # normal wells
				else:
					well.setType(2)    # control wells


				# if the plate was provided, set the tooltip to display:
				#   wellID
				#   gene symbol
				#   gene description
				#   score
				if plate:
					wellStr = '%s%s<br>' % (row, col)

					if vPlate.well[row][col].symbol != None:
						#geneStr = '%s<br>' % vPlate.vSymbol[row][col]
						geneStr = '%s<br>' % vPlate.well[row][col].symbol
					else:
						geneStr = '(No symbol)<br>'

					if vPlate.well[row][col].prefName != None and vPlate.well[row][col].prefName != '\N':
						descStr = '%s<br>' % vPlate.well[row][col].prefName
					else:
						descStr = ''

					try:
						scoreStr = '%9.3g' % plateData[row][col]
					except:
						scoreStr = ''

					well.setToolTipText('<html>%s%s%s%s</html>' % (wellStr, geneStr, descStr, scoreStr))
					#well.setToolTipText('%s' % self.colorScale(val))

		return (minVal, maxVal)

	def stateChanged(self, evt):
		# set self.minThresh, maxThresh, then repaint
		scaledMin = evt.getSource().getScaledMin()
		scaledMax = evt.getSource().getScaledMax()
		if scaledMin == None or scaledMax == None:
			self.minThresh = 0.0
			self.maxThresh = 1.0
		else:
			self.minThresh = 1.0 - scaledMin
			self.maxThresh = 1.0 - scaledMax
		self.paintWells(normColor=1)
		self.repaint()

class JWellPanel(JPanel):
	# each well is displayed as its own JPanel
	def __init__(self, cShape):
		JPanel.__init__(self) #@UndefinedVariable
		self.cType = 1
		self.bgColor = Color.WHITE
		self.fgColor = COLOR_UNDEFINED
		self.lCoord1 = None
		self.cDiam = None
		self.cShape = cShape

	def setType(self, cType):
		self.cType = cType
		if self.lCoord1 == None:
			self.x1 = 7.0            # center of the well
			self.y1 = self.x1 - 1    # one pixel below...
			self.y2 = self.x1 + 1    # ...and one above (for EMPTY mark)

	def setFgColor(self, color):
		# well foreground color
		self.fgColor = color

	def setBgColor(self, color):
		# well background color
		self.bgColor = color

	def setWellState(self, fg, cType):
		# set the state of the well (and foreground color for next repaint)
		self.fgColor = fg
		self.cType = cType

	def paintComponent(self, g):
		# repaint the well
		self.super__paintComponent(g)
		if self.cType == 1:					# normal hairpin
			self.setBackground(self.fgColor)
		elif self.cType == 0:				  # empty
			self.setBackground(self.fgColor)
			g.setColor(Color.BLACK)
			originalStroke = g.getStroke()
			g.setStroke(BasicStroke(2.0))
			g.draw(LineDouble(self.x1, self.y1, self.x1, self.y2))   # EMPTY mark
			g.setStroke(originalStroke)

		else:								# control/pgw
			self.setBackground(self.bgColor)
			g.setColor(self.fgColor)
			g.fill(self.cShape)

class ColorScalePanel(JPanel, java.awt.event.MouseListener, java.awt.event.MouseMotionListener):
	# color scale, which can also be used to set the min/max values to display
	# (used next to the two plate maps on the Plate Scoring tab

	# the triangle-shaped limit pointer
	triXpoints = [1, 6, 6]
	triYpoints = [0, 5, -5]

	def __init__(self, colorScale=_redGreen, minVal=0.0, maxVal=1.0, thickness=15, vertical=1):
		self.minVal = minVal
		self.minSliderVal = minVal
		self.maxVal = maxVal
		self.maxSliderVal = maxVal
		self.vertical = vertical
		self.thickness = thickness
		self.colorScale = colorScale
		self.margin = 5
		self.changeListeners = []

		# defaults
		self.isGrabbed = 0
		self.isMaxGrabbed = 0
		self.clickOffset = 0

		# create triangle pointers for limit setting 
		self.minSlider = Polygon(self.triXpoints, self.triYpoints, len(self.triXpoints))
		self.maxSlider = Polygon(self.triXpoints, self.triYpoints, len(self.triXpoints))

		# add the listeners
		self.addMouseListener(self)
		self.mouseListenerList = self.getMouseListeners()
		self.addMouseMotionListener(self)
		self.mouseMotionListenerList = self.getMouseMotionListeners()

	def addChangeListener(self, listener):
		# add a change listener
		self.changeListeners.append(listener)

	def setRange(self, minVal, maxVal):
		# set the min/max of the color scale
		self.minVal = minVal
		self.minSliderVal = minVal
		self.maxVal = maxVal
		self.maxSliderVal = maxVal
		# send change event when mouse changes:
		changeEvent = ChangeEvent(self)
		for eachListener in self.changeListeners:
			eachListener.stateChanged(changeEvent)

		self.repaint()

	def setColorScale(self, colorScale):
		# allows the color scale to be set as red/green, spectrum, or blue/yellow
		self.colorScale = colorScale

	def getScaledMin(self):
		# get the scaled minimum value
		if self.minVal == None or self.maxVal == None or self.maxVal - self.minVal == 0:
			return None
		else:
			return (self.maxVal - self.minSliderVal) / (self.maxVal - self.minVal)

	def getScaledMax(self):
		# get the scaled maximum value
		if self.minVal == None or self.maxVal == None or self.maxVal - self.minVal == 0:
			return None
		else:
			return (self.maxVal - self.maxSliderVal) / (self.maxVal - self.minVal)

	def paintComponent(self, g):
		# repaint the color scale
		self.super__paintComponent(g)
		if self.vertical:
			barHeight = float(self.getSize().height - 2 * self.margin)

			arrowX = self.margin + self.thickness + 1

			scaledMin = self.getScaledMin()
			scaledMax = self.getScaledMax()
			if scaledMin == None:
				arrowYmin = int(0.0 * barHeight + self.margin)
			else:
				arrowYmin = int(scaledMin * barHeight + self.margin)

			if scaledMax == None:
				arrowYmax = int(1.0 * barHeight + self.margin)
			else:
				arrowYmax = int(scaledMax * barHeight + self.margin)

			self.minSlider.xpoints = self.triXpoints
			self.minSlider.ypoints = self.triYpoints
			self.maxSlider.xpoints = self.triXpoints
			self.maxSlider.ypoints = self.triYpoints

			i = self.margin


			g.setColor(HIGH_COLOR)   # high color

			# TEST: Draw 3 lines above the 'margin' of HIGH_COLOR:
			g.drawLine(self.margin, i - 3, self.thickness + self.margin, i - 3)
			g.drawLine(self.margin, i - 2, self.thickness + self.margin, i - 2)
			g.drawLine(self.margin, i - 1, self.thickness + self.margin, i - 1)

			while i < arrowYmax:
				g.drawLine(self.margin, i, self.thickness + self.margin, i)
				i += 1

			height = self.getSize().height
			while i < arrowYmin:
				g.setColor(self.colorScale(1.0 - (i - self.margin) / barHeight))
				g.drawLine(self.margin, i, self.thickness + self.margin, i)
				i += 1

			g.setColor(LOW_COLOR)   # low color
			while i < height - self.margin:
				g.drawLine(self.margin, i, self.thickness + self.margin, i)
				i += 1

			# TEST: Draw 3 lines below the 'margin' of LOW_COLOR:
			g.drawLine(self.margin, i, self.thickness + self.margin, i)
			g.drawLine(self.margin, i + 1, self.thickness + self.margin, i + 1)
			g.drawLine(self.margin, i + 2, self.thickness + self.margin, i + 2)

			# fill the triangles with black
			g.setColor(Color.BLACK)
			self.minSlider.translate(arrowX, arrowYmin - 1)
			g.fillPolygon(self.minSlider)
			self.maxSlider.translate(arrowX, arrowYmax)
			g.fillPolygon(self.maxSlider)

			if self.maxSliderVal == None:
				# if htere is no "max slider value", remove the listeners and change the slider label to N/A
				for listener in self.getMouseListeners():
					self.removeMouseListener(listener)
				for listener in self.getMouseMotionListeners():
					self.removeMouseMotionListener(listener)
				g.drawString("N/A", 2 * self.margin + self.thickness + 4,
							 self.maxSlider.ypoints[0] + g.getFontMetrics().getAscent() / 2 - 1)
			else:
				# if the listeners have been removed (because an empty plate was displayed, etc.), add the 
				# listeners back, and re-label the slider arrow
				if len(self.getMouseListeners()) == 0:
					for listener in self.mouseListenerList:
						self.addMouseListener(listener)
				if len(self.getMouseMotionListeners()) == 0:
					for listener in self.mouseMotionListenerList:
						self.addMouseMotionListener(listener)

				g.drawString("%s" % str(self.maxSliderVal), 2 * self.margin + self.thickness + 4,
							 self.maxSlider.ypoints[0] + g.getFontMetrics().getAscent() / 2 - 1)

			# same as for the maxSliderVal (above), but the listeners have already been removed or re-attached
			if self.minSliderVal == None:
				g.drawString("N/A", 2 * self.margin + self.thickness + 4,
							 self.minSlider.ypoints[0] + g.getFontMetrics().getAscent() / 2 - 1)
			else:
				g.drawString("%s" % str(self.minSliderVal), 2 * self.margin + self.thickness + 4,
							 self.minSlider.ypoints[0] + g.getFontMetrics().getAscent() / 2 - 1)


	# overridden from MouseListener
	def mousePressed(self, event):
		"""Called when mouse is clicked and held down. Initializes the slider
		dragging environment."""
		self.minSlider.invalidate()
		hitBox = self.minSlider.getBounds()
		hitBox.grow(3, 3)
		if hitBox.contains(event.x, event.y):
			self.isGrabbed = 1
			self.isMaxGrabbed = 0
			self.clickOffset = self.minSlider.ypoints[0] - event.y
		else:
			self.maxSlider.invalidate()
			hitBox = self.maxSlider.getBounds()
			hitBox.grow(3, 3)
			if hitBox.contains(event.x, event.y):
				self.isGrabbed = 1
				self.isMaxGrabbed = 1
				self.clickOffset = self.maxSlider.ypoints[0] - event.y
			else:
				self.isGrabbed = 0

	def mouseReleased(self, event):
		self.isGrabbed = 0
		# update when mouse released
		changeEvent = ChangeEvent(self)
		for eachListener in self.changeListeners:
			eachListener.stateChanged(changeEvent)

	# overridden from MouseMotionListener
	def mouseDragged(self, event):
		barHeight = float(self.getSize().height - 2 * self.margin)

		if self.isGrabbed == 0:
			return

		arrowY = event.y + self.clickOffset

		if self.isMaxGrabbed:
			if arrowY > self.minSlider.ypoints[0]:
				arrowY = self.minSlider.ypoints[0]
			elif arrowY < self.margin:
				arrowY = self.margin

			self.maxSliderVal = self.maxVal - (arrowY - self.margin) * (self.maxVal - self.minVal) / barHeight
		else:
			if arrowY < self.maxSlider.ypoints[0]:
				arrowY = self.maxSlider.ypoints[0]
			elif arrowY > self.getSize().height - self.margin:
				arrowY = self.getSize().height - self.margin

			self.minSliderVal = self.maxVal - (arrowY - self.margin) * (self.maxVal - self.minVal) / barHeight

		# send change event when mouse changes:
		changeEvent = ChangeEvent(self)
		for eachListener in self.changeListeners:
			eachListener.stateChanged(changeEvent)

		self.repaint()

	# ignore other mouse events
	def mouseEntered(self, event):
		pass

	def mouseExited(self, event):
		pass

	def mouseClicked(self, event):
		pass

	def mouseMoved(self, event):
		pass

class MessagePopup(JOptionPane):
	# use a JOptionPane for pop-up confirm dialog box
	def __init__(self, parent):
		JOptionPane.__init__(self) #@UndefinedVariable
		self.parentFrame = parent

	# the default message type is a warning message:
	def showParentedConfirmDialog(self, string, title, mType=JOptionPane.WARNING_MESSAGE):
		response = self.showConfirmDialog(self.parentFrame, string, title,
										  JOptionPane.OK_CANCEL_OPTION, mType)
		return response

	def showParentedMessageDialog(self, string, title, mType=JOptionPane.WARNING_MESSAGE):
		self.showMessageDialog(self.parentFrame, string, title, mType)
		return

class DefaultGridbagConstraints(java.awt.GridBagConstraints):
	# explicit default GridBagConstraints:
	def __init__(self):
#		GridBagConstraints.__init__(self) #@UndefinedVariable
		self.gridx = 0
		self.gridy = 0
		self.gridwidth = 1
		self.gridheight = 1
		self.weightx = 1.0
		self.weighty = 1.0
		self.anchor = GridBagConstraints.CENTER
		self.fill = GridBagConstraints.BOTH
		self.insets = Insets(0, 0, 0, 0)
		self.ipadx = 0
		self.ipady = 0

	def resetDefaults(self):
		self.gridx = 1
		self.gridy = 1
		self.gridwidth = 1
		self.gridheight = 1
		self.weightx = 1.0
		self.weighty = 1.0
		self.anchor = GridBagConstraints.CENTER
		self.fill = GridBagConstraints.BOTH
		self.insets = Insets(0, 0, 0, 0)
		self.ipadx = 0
		self.ipady = 0


class LabelledProgressBar(JPanel):
	# a progress bar with labeling
	def __init__(self, invisibleColor=Color.newColor(238, 238, 238)):
		JPanel.__init__(self, GridLayout(1, 2, 5, 0)) #@UndefinedVariable
		self.text = JLabel()
		self.text.setHorizontalAlignment(JLabel.RIGHT)
		self.add(self.text)
		# the progress bar is a simple JProgressBar component
		self.progressBar = JProgressBar(0, 100)
		self.add(self.progressBar)
		self.invisible = invisibleColor

	def setIndeterminate(self, state):
		# Indeterminate state (position in progress is unknown)
		self.progressBar.setIndeterminate(state)
		self.progressBar.paint(self.progressBar.getGraphics())

	def setProgress(self, pct):
		# set the percent of progress
		self.progressBar.setValue(int(pct * 100.0))
		self.progressBar.paint(self.progressBar.getGraphics())

	def setLabelText(self, text):
		# set progress bar label
		g = self.text.getGraphics()
		self.text.setText(text)
		if g != None:
			self.update(g)

	def setVisibility(self, visible):
		# set visibility
		if visible == 1:
			self.text.setVisible(1)
			self.progressBar.setVisible(1)
		else:
			self.text.setVisible(0)
			self.progressBar.setVisible(0)

		if self.text.getGraphics() != None:
			self.text.paint(self.text.getGraphics())
			self.progressBar.paint(self.progressBar.getGraphics())


def chooseFile():
	# basic file chooser
	outFileName = None
	# open in the default directory, unless the directory has already been set:
	if masterObject.getField('fileDirectory') == None:
		fc = JFileChooser()
	else:
		fc = JFileChooser(masterObject.getField('fileDirectory'))

	# get the file filter:
	fileFilter = masterObject.getField('fileFilter')
	fc.setFileFilter(fileFilter)

	# "save file" dialog
	fc.setDialogType(JFileChooser.SAVE_DIALOG)
	fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES)
	mw = masterObject.getField('topWindow')
	retVal = fc.showSaveDialog(mw)
	masterObject.setField('fileDirectory', fc.getCurrentDirectory())
	if retVal == JFileChooser.APPROVE_OPTION:
		outFile = fc.getSelectedFile()
		outFileName = outFile.getAbsolutePath()
		if os.path.splitext(outFileName)[1] == '':
			outFileName = '%s.tab' % outFileName
		print 'Output file: %s' % outFileName
	else:
		print 'Selection cancelled by user'

	return openResultsFile(outFileName)

# for RIGER/GENE-E file output - only .gct and .chip files
def chooseGCTFile():
	outFileName = None
	# open in the default directory, unless the directory has already been set:
	if masterObject.getField('fileDirectory') == None:
		fc = JFileChooser()
	else:
		fc = JFileChooser(masterObject.getField('fileDirectory'))

	# get the file filter:
	fileFilter = masterObject.getField('gctFilter')
	fc.setFileFilter(fileFilter)

	fc.setDialogType(JFileChooser.SAVE_DIALOG)
	fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES)
	mw = masterObject.getField('topWindow')
	retVal = fc.showSaveDialog(mw)
	masterObject.setField('fileDirectory', fc.getCurrentDirectory())
	if retVal == JFileChooser.APPROVE_OPTION:
		outFile = fc.getSelectedFile()
		gctFileName = outFile.getAbsolutePath()
		gctFileName = '%s.gct' % os.path.splitext(gctFileName)[0]
		chipFileName = '%s.chip' % os.path.splitext(gctFileName)[0]
		print 'Output files: %s, %s' % (gctFileName, chipFileName)
	else:
		print 'Selection cancelled by user'

	## TG: seems like these should be moved up into the scope of the above "if"...
	## AD: If so, fGct and fChip should be initialized to "None"
	fGct = openResultsFile(gctFileName)
	fChip = openResultsFile(chipFileName)
	return (fGct, fChip)

def chooseInputFile():
	# input file chooser
	outFileName = None     # AD: should probably be renamed 'inFileName'
	# open in the default directory, unless the directory has already been set:
	if masterObject.getField('fileDirectory') == None:
		fc = JFileChooser()
	else:
		fc = JFileChooser(masterObject.getField('fileDirectory'))

	# get the file filter:
	fileFilter = masterObject.getField('fileFilter')
	fc.setFileFilter(fileFilter)

	fc.setDialogType(JFileChooser.OPEN_DIALOG)
	fc.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES)
	mw = masterObject.getField('topWindow')
	retVal = fc.showOpenDialog(mw)
	masterObject.setField('fileDirectory', fc.getCurrentDirectory())
	if retVal == JFileChooser.APPROVE_OPTION:
		outFile = fc.getSelectedFile()
		outFileName = outFile.getAbsolutePath()
	else:
		print 'Selection cancelled by user'

	#return openResultsFile(outFileName)
	return outFileName

def openResultsFile(fileName):
	fout = None
	if fileName != None:
		# open the output file
		try:
                        fout = codecs.open(fileName, 'w', encoding='utf-8')
		except IOError, emsg:
			popup = masterObject.getField('errorPopup')
			response = popup.showParentedConfirmDialog(emsg.args[0], 'Error opening file')
			# 
			if response == JOptionPane.OK_OPTION:
				# try a different filename:
				fout = chooseFile()
			# otherwise, CANCEL chosen, return None:
		except:
			popup = masterObject.getField('errorPopup')
			popup.showParentedMessageDialog('Some non-IOError occurred', 'Error opening file')

	return fout    # AD: Out-dented by one level (was inside the "if"

def printToLog(text):
	if masterObject.getField('debugging'):
		print text

