from DbLoader import loadPlateSetFromDb
from javawrap import GridBagConstraints
from rnaeyes import MainHandle
import MajorObjects
import PlateSet
import java.awt.Color as Color
import java.awt.GridBagLayout as GridBagLayout
import java.awt.Insets as Insets
import javax.swing as swing
import javax.swing.JPanel as JPanel

# get the masterObject for global variables:
masterObject = MainHandle.masterObject

# same borders for all top-level subpanels:
loweredEtched = swing.BorderFactory.createCompoundBorder(swing.BorderFactory.createEmptyBorder(1, 1, 1, 1),
														 swing.BorderFactory.createEtchedBorder(swing.border.EtchedBorder.LOWERED))

INVISIBLE_COLOR = Color(238, 238, 238)

def projectSelect_callback(listBox):
	# callback invoked when a project is selected from the list of available projects
	# (disable plate set load button until screen(s) and batch(es) are selected)
	masterObject.getField('loadPlateSetButton').setEnabled(0)

	# get the user name and their authority. 
	#	If the authority is 'screener' they will only be able to see data that they are authorized to see.
	#	If the authority is 'all' they will be able to see ALL screening data
	user = masterObject.getField('uName')
	authority = masterObject.getField('authority')

	# get the indices of the selected projects from the project list
	myList = listBox.get_myList()                  # actual swing JList object
	projects = myList.getSelectedValues()          # get the selected project(s)
	projectIdxList = myList.getSelectedIndices()   # get the indices of the selected items
	projectIdList = None
	for pid in projectIdxList:
		if projectIdList == None:
			projectIdList = "('%s'" % listBox.get_myDataList()[pid]
		else:
			projectIdList = "%s,'%s'" % (projectIdList, listBox.get_myDataList()[pid])
	projectIdList = '%s)' % projectIdList

	# clear selections from the "Available Screens:" and "Batch list:" lists and plate list display:
	parent = listBox.myContainer       # the parent window contains all of the lists (project, screen, batch, and plate)
	screenListbox = parent.compDict['screenList']
	screenListbox.setData([], [])
	batchListbox = parent.compDict['batchList']
	batchListbox.setData([], [])
	plateListbox = parent.compDict['plateList']
	plateListbox.setData([], [])

	# get the connector handle:
	conn = masterObject.getField('conn')

	# get the list of all screens for this project (for which the user has viewing authority):
	if authority == 'all':
		# unrestricted access
		qryStr = ('select screenName, screenId from SCREEN where projectId in %s order by upper(screenname)' % projectIdList)
		#print qryStr
		sQry = conn.prepareStatement(qryStr)
	else:
		# restricted by entries in PERSON2SCREEN
		qryStr = ('select screenName, screenId from SCREEN where projectId in %s and screenId in (select screenId from person2screen p2s, person p where p.loginname=? and p.PERSONID=p2s.PERSONID) order by upper(screenname)' % projectIdList)
		#print qryStr
		sQry = conn.prepareStatement(qryStr)
		sQry.setString(1, user)

	# get the list of all screen names and screen IDs available in the database for the select project(s):
	rs = sQry.executeQuery()
	sList = []
	sIdList = []
	# create a list of screen names and screen IDs returned from the DB:
	while rs.next():
		sList.append(rs.getString(1))     # screen names
		sIdList.append(rs.getString(2))   # screen IDs

	# update the list of screens under the "Available Screens:" column header:
	screenListbox.setData(sList, sIdList)

	# set/reset master object project field:
	masterObject.setField('projectName', projects)
	masterObject.setField('projectId', projectIdList)

def screenSelect_callback(listBox):
	# callback invoked when a screen is selected from the list of available screens:
	masterObject.getField('loadPlateSetButton').setEnabled(0)  # disable plate set load button
	myList = listBox.get_myList()              # actual swing JList object
	screen = myList.getSelectedValues()        # get the selected screen(s)
	screenIdx = myList.getSelectedIndices()    # get the indices of the selected screen(s)
	screenIdList = listBox.get_myDataList()    # get the list of all available screenIDs

	# same process as project selection above
	selectedScreenList = ''
	selectedScreenSet = []
	for i in range(len(screen)):
		selectedScreenList = "%s'%s'," % (selectedScreenList, screenIdList[screenIdx[i]])
		selectedScreenSet.append(screenIdList[screenIdx[i]])

	selectedScreenList = '(%s)' % selectedScreenList
	selectedScreenList = selectedScreenList.replace(',)', ')')  # remove final comma from the list string

	parent = listBox.myContainer

	# clear selections from other lists:
	batchListbox = parent.compDict['batchList']
	batchListbox.setData([], [])
	plateListbox = parent.compDict['plateList']
	plateListbox.setData([], [])

	# update the selectedScreen global and clear the selectedBatches and selectedPlates:
	masterObject.setField('selectedScreens', selectedScreenSet)
	masterObject.setField('selectedBatches', [])
	masterObject.setField('selectedPlates', [])

	# get the connector handle:
	conn = masterObject.getField('conn')

	# get the list of all screens for the selected project(s):
	sFields = 'i.infectionBatchName, i.infectionBatchId, s.screenName'
	qryStr = ('select %s from INFECTIONBATCH i, SCREEN s where i.screenId in %s and s.screenId=i.screenId order by upper(infectionbatchname)' % (sFields, selectedScreenList))

	bQry = conn.prepareStatement(qryStr)
	rs = bQry.executeQuery()
	bList = []
	bIdList = []
	batch2screen = {}
	while rs.next():
		bList.append(rs.getString(1))                                 # infection batch name
		bIdList.append(rs.getString(2))                               # infection batch ID
		batch2screen.setdefault(rs.getString(1), rs.getString(3))     # infection batch: screen name mapping

	masterObject.setField('batch2screen', batch2screen)    # update batch2screen global map

	# update the list of screens under the "Batch list:" column header:
	batchListbox.setData(bList, bIdList)

def batchSelect_callback(listBox):
	# callback invoked when a batch is selected from the list of available batchess:
	masterObject.getField('loadPlateSetButton').setEnabled(0)  # disable plate set load button
	myList = listBox.get_myList()                  # actual swing JList object
	batch = myList.getSelectedValues()             # get the selected batch(es)
	batchIdx = myList.getSelectedIndices()         # get the indices of the selected batches
	batchIdList = listBox.get_myDataList()         # get the full list of batch IDs

	# same process as screen and project selection above
	selectedBatchList = ''
	selectedBatchSet = []
	for i in range(len(batch)):
		selectedBatchList = "%s'%s'," % (selectedBatchList, batchIdList[batchIdx[i]])
		selectedBatchSet.append(batchIdList[batchIdx[i]])

	selectedBatchList = '(%s)' % selectedBatchList
	selectedBatchList = selectedBatchList.replace(',)', ')')    # remove last comma from the batch list string
	#print selectedBatchList

	parent = listBox.myContainer
	# clear the list of available plates:
	plateListbox = parent.compDict['plateList']
	plateListbox.setData([], [])
	masterObject.setField('selectedBatches', selectedBatchSet)  # update global list of batches

	# get the connector handle:
	conn = masterObject.getField('conn')

	# get the list of all plates for the selected screen(s) and batch(es):
        qryStr = '''
SELECT
  p.screenerPlateName, p.plateId, ib.infectionBatchName, ib.infectionBatchId, s.screenName, s.screenId, ar.assayReadoutId, count(1)
FROM infectionbatch ib, screen s, infection i, assayreadout ar, plate p, infectionwell iw
WHERE ib.infectionBatchId IN %s
AND ib.screenId = s.screenId
AND ib.infectionbatchid = i.infectionbatchid
AND i.infectionid = ar.infectionid
AND i.plateid = p.plateid
AND p.plateid = iw.infectionplateid
GROUP BY
  p.screenerPlateName, p.plateId, ib.infectionBatchName, ib.infectionBatchId, s.screenName, s.screenId, ar.assayReadoutId
ORDER BY s.screenName, ib.infectionBatchName, p.screenerplateName
''' % selectedBatchList

	pQry = conn.prepareStatement(qryStr)
	rs = pQry.executeQuery()
	pStringList = []
	pList = []
	pIdList = []
	bList = []
	bIdList = []
	sList = []
	sIdList = []
	rIdList = []
	nWellList = []

	# read the result of database query and append each data field to the appropriate list.
	# The lists are designed to be parallel for easy display in tables.
	while rs.next():
		plateName = rs.getString(1)

                # for cases where there is a ":" in a plate name, replace it with a "_" for display in any list within RNAeyes
                plateName = plateName.replace(':', '_')

		# add the plate to the list of available plates
		pList.append(plateName)
		pIdList.append(rs.getString(2))       # plateId       
		batchName = rs.getString(3)           # infectionBatchName
		bList.append(batchName)
		bIdList.append(rs.getString(4))       # infectionBatchId
		screenName = rs.getString(5)
		sList.append(screenName)
		sIdList.append(rs.getString(6))       # screenId
		pString = '%s : %s : %s' % (screenName, batchName, plateName)
		pStringList.append(pString)
		rIdList.append(rs.getString(7))       # assayId
		nWellList.append(rs.getInt(8))        # wellCount

	#plateListbox.setData(pList, pIdList)
	plateListbox.setData(pStringList, pIdList)	  # screenName : batchName : plateName -> pId 

	# store selected screen/batch/plate info:
	plateListbox.setField('pList', pList)			   #
	plateListbox.setField('pIdList', pIdList)		   #
	plateListbox.setField('bList', bList)			   #
	plateListbox.setField('bIdList', bIdList)		   #
	plateListbox.setField('sList', sList)			   #
	plateListbox.setField('sIdList', sIdList)		   #

	masterObject.setField('readIdList', rIdList)		 #
	masterObject.setField('nWellList', nWellList)		#
	masterObject.setField('plateNameList', pList)		#
	masterObject.setField('plateIdList', pIdList)		#
	masterObject.setField('screenNameList', sList)	   #
	masterObject.setField('screenIdList', sIdList)	   #
	masterObject.setField('batchNameList', bList)		#
	masterObject.setField('batchIdList', bIdList)		#

	masterObject.getField('loadPlateSetButton').setEnabled(1)  # enable plate set load button

def createPlateSetSelectionPanel(userName, resetPwButton):
	conn = masterObject.getField('conn')

	# create main frame:
	plateSetSelPanel = MajorObjects.NewPanel()
	plateSetSelPanel.setLayout(GridBagLayout())

	###################################################################################
	#  Plate selection mode area layout
	###################################################################################
	plateSetModePanel = makeModeSelectPanel()

	###################################################################################
	#  Plate set selection from DB panel
	###################################################################################
	(plateSetListPanel, compList) = makePlateSetSelectionPanel()
	MainHandle.addToMasterObject(compList)


	###################################################################################
	#  Load data button and progress bars
	###################################################################################
	(plateLoadingPanel, compList) = makePlateLoadingPanel()
	MainHandle.addToMasterObject(compList)

	plateSetDbPanel = masterObject.getField('plateSetDbPanel')
	buttonPane = masterObject.getField('loadPlateSetButtonPane')
	buttonPane.addKeyValuePair('sourceData', plateSetDbPanel)
	projectList = masterObject.getField('projectList')
	plateSetSelPanel.addComponentId('projectList', projectList)
	screenList = masterObject.getField('screenList')
	plateSetSelPanel.addComponentId('screenList', screenList)
	batchList = masterObject.getField('batchList')
	plateSetSelPanel.addComponentId('batchList', batchList)
	plateList = masterObject.getField('plateList')
	plateSetSelPanel.addComponentId('plateList', plateList)
	plateSetListPanel = masterObject.getField('plateSetListPanel')
	plateSetSelPanel.addComponentId('listPanel', plateSetListPanel)
	loadPlateSetButton = masterObject.getField('loadPlateSetButton')
	plateSetSelPanel.addComponentId('loadPlateSetButton', loadPlateSetButton)

	###################################################################################
	#  file/pickle-based plate selection area layout
	###################################################################################
	plateSetFilePanel = MajorObjects.NewPanel()	  # for file/pickle-based plate selection
	plateSetFilePanel.setBorder(loweredEtched)

	# initialize by loading the list of all available projects:
	qryStr = ('select distinct p.projectName, p.projectId from PROJECT p, SCREEN s where p.parentProjectId is not null and p.projectid=s.projectid order by upper(projectname)')
	pQry = conn.prepareStatement(qryStr)
	rs = pQry.executeQuery()
	pList = []
	pIdList = []
	while rs.next():
		pList.append(rs.getString(1))
		pIdList.append(rs.getString(2))

	projectList.setData(pList, pIdList)

	# GridBagConstraints parameters:
	tConstraints = GridBagConstraints.newGridBagConstraints()
	tConstraints.gridx = 0
	tConstraints.gridwidth = 1
	tConstraints.gridheight = 4
	tConstraints.weightx = 1.0
	tConstraints.weighty = 0.31
	tConstraints.anchor = GridBagConstraints.NORTH
	tConstraints.fill = GridBagConstraints.BOTH
	tConstraints.ipadx = 0
	tConstraints.ipady = 0

	# reset password panel:
	resetPW = JPanel()
	resetPW.add(resetPwButton)

	# add all components to the Plate Set Selection Panel:
	plateSetSelPanel.addComponent('plateSetModePanel', plateSetModePanel, tConstraints)

	plateSetSelPanel.addComponent('plateSetListPanel', plateSetListPanel, tConstraints)
	tConstraints.anchor = GridBagConstraints.NORTHEAST
	tConstraints.fill = GridBagConstraints.HORIZONTAL

	plateSetSelPanel.addComponent('plateLoadingPanel', plateLoadingPanel, tConstraints)
	tConstraints.weighty = 0.05
	tConstraints.anchor = GridBagConstraints.CENTER
	plateSetSelPanel.addComponent('resetPwPanel', resetPW, tConstraints)

	return plateSetSelPanel

def makeModeSelectPanel():
	dataLoadingModePanel = JPanel()
	dataLoadingModePanel.setBorder(loweredEtched)
	return dataLoadingModePanel

def makePlateSetSelectionPanel():

	###################################################################################
	#  DB-based plate selection area layout
	###################################################################################
	plateSetDbPanel = MajorObjects.NewPanel()		# for DB-based plate selection
	plateSetDbPanel.setBorder(loweredEtched)
	plateSetDbPanel.setLayout(swing.BoxLayout(plateSetDbPanel, swing.BoxLayout.Y_AXIS))

	# Project/Screen/Batch/Plate listboxes: #################################
	plateSetListPanel = MajorObjects.NewPanel()
	#plateSetListPanel.setLayout(swing.BoxLayout(plateSetListPanel,swing.BoxLayout.X_AXIS))
	plateSetListPanel.setLayout(GridBagLayout())

	# project ListBox:
	projectList = MajorObjects.ListBox('Project List:', (150, 200), swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION)
	projectList.setData([], [])
	projectList.setWidthByPrototype("XXXXXXXXXXXXXXXXXXXXXXXXXXX")
	projectList.setHolder(plateSetListPanel)
	projectList.addListSelectCallback(projectSelect_callback)

	# screen ListBox:
	screenList = MajorObjects.ListBox('Available Screens:', (100, 200), swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION)
	screenList.setData([], [])
	screenList.setWidthByPrototype("XXXXXXXXXXXXXXXXXXXXXXXX")
	screenList.setHolder(plateSetListPanel)
	screenList.addListSelectCallback(screenSelect_callback)

	# batch ListBox:
	batchList = MajorObjects.ListBox('Batch list:', (100, 200), swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION)
	batchList.setData([], [])
	batchList.setWidthByPrototype("XXXXXXXXXXXXXXXXXXX")
	batchList.setHolder(plateSetListPanel)
	batchList.addListSelectCallback(batchSelect_callback)

	# plate ListBox:
	plateList = MajorObjects.ListBox('Chosen plate list (Screen : Batch : Plate):', (150, 200), swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION)
	plateList.setData([], [])
	plateList.setWidthByPrototype("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXxxxxxxxxxxxxxxxx")

	# add the list components to the panel, and add their identifiers to the top-level panel:
	pConstraints = MajorObjects.DefaultGridbagConstraints()
	pConstraints.fill = GridBagConstraints.BOTH
	plateSetListPanel.addComponent('projectList', projectList, pConstraints)
	pConstraints.gridx = GridBagConstraints.RELATIVE
	plateSetListPanel.addComponent('screenList', screenList, pConstraints)
	plateSetListPanel.addComponent('batchList', batchList, pConstraints)
	plateSetListPanel.addComponent('plateList', plateList, pConstraints)

	pConstraints = MajorObjects.DefaultGridbagConstraints()
	pConstraints.fill = GridBagConstraints.BOTH
	pConstraints.gridy = 0
	pConstraints.gridheight = 8

	plateSetDbPanel.add(plateSetListPanel, pConstraints)

	return (plateSetListPanel, [['projectList', projectList],
								['screenList', screenList],
								['batchList', batchList],
								['plateList', plateList],
								['plateSetListPanel', plateSetListPanel],
								['plateSetDbPanel', plateSetDbPanel]])

def makePlateLoadingPanel():
	# panel containing the "Load Plate Set Data" button
	plateLoadingPanel = JPanel(GridBagLayout())
	bConstraints = MajorObjects.DefaultGridbagConstraints()
	bConstraints.fill = GridBagConstraints.BOTH
	bConstraints.anchor = GridBagConstraints.EAST
	bConstraints.insets = Insets(2, 2, 2, 2)
	bConstraints.gridx = GridBagConstraints.RELATIVE

	bConstraints.weightx = 8.0
	dummy1 = JPanel()
	dummy2 = JPanel()
	plateLoadingPanel.add(dummy1, bConstraints)
	plateLoadingPanel.add(dummy2, bConstraints)
	bConstraints.weightx = 1.0

	# create and add the 'Load Plate Data' button:
	buttonPane = MajorObjects.BorderedButton('Load Plate Set Data', loadPlateSetFromDb, layout='gridBag')
	buttonPane.button.setEnabled(0)  # initialize to disabled
	bConstraints.fill = GridBagConstraints.NONE
	bConstraints.anchor = GridBagConstraints.EAST
	bConstraints.weightx = 0.25
	plateLoadingPanel.add(buttonPane, bConstraints)

	loadingProgressBar = MajorObjects.LabelledProgressBar()
	loadingProgressBar.setVisibility(0)
	loadingProgressBar.progressBar.setForeground(Color(255, 100, 100))
	plateLoadingPanel.add(loadingProgressBar, bConstraints)

	return (plateLoadingPanel, [['loadPlateSetButton', buttonPane.button],
								['loadPlateSetButtonPane', buttonPane],
								['loadingProgressBar', loadingProgressBar]])

def makePasswordResetPanel():
	# this is apparently not used
	pass

def getIePlates(plateSet):
	# build the lists of plates that can be used for infection efficiency calculation
	plateIdList = plateSet.getPlateIdList()    # all plates
	iePlates = PlateSet('iePlates')            # create a plateset for only the IE plates
	vPlateGroups = {}

	# make a PlateSet of the infection efficiency plates:
	for plateId in plateIdList:
		plate = plateSet.getPlate(plateId)
		if not plate.use:
			# plates have a "use" flag that indicates whether or not they should be used for infection efficiency calculation
			continue
		if plate.puroRep == 'YES' and \
			   (plate.cellViability == None or plate.cellViability.upper() == 'YES'):
			# only add the plate to the set if:
			#	- it is considered a "replicate"
			#	- the cell viability flag is null or (case-insensitive) "yes"
			iePlates.addPlate(plate)

		vId = '%s;%s' % (plate.vPlateId, plate.batchId)       # virus plate ID = plateID:batchID
		vName = '%s;%s' % (plate.vPlateName, plate.batchName) # virus plate name = plate name:batch name (for display)
		puro = plate.selStatus.upper()    # get the puro selection status
		if plate.cellViability == None or plate.cellViability.upper() == 'YES':
			# AD: this shoul already be determined when it was added to the list of iePlates
			viabilityScreen = 1
		else:
			viabilityScreen = 0  # AD: this should never happen

		# group the plates by virus plate ID (vId) and puro status (puro+ and puro-)
		vPlateGroups.setdefault(vId, {})
		vPlateGroups[vId].setdefault('vpName', vName) # name the group by the virus plate name (vName)
		vPlateGroups[vId].setdefault('puro+', [])     # puro+ plates
		vPlateGroups[vId].setdefault('puro-', [])     # puro- plates
		if (puro == 'YES' or puro == 'PLUS') and plate.puroRep == 'YES' and viabilityScreen:
			# puro+
			vPlateGroups[vId]['puro+'].append(plateId)
		elif (puro == 'NO' or puro == 'MINUS') and plate.puroRep == 'YES' and viabilityScreen:
			# puro-
			vPlateGroups[vId]['puro-'].append(plateId)

	# add the vPlateGroups map to the iePlates plateset:
	iePlates.addKeyValuePair('vIePlateGroups', vPlateGroups)
	return iePlates

def averagePlateSet(plateSet):
	# first, check to make sure this is a valid IE plateset:
	vPlateGroups = plateSet.getValue('vIePlateGroups')
	if not vPlateGroups:
		return None

	# next, loop over all virus plates, and average the sets of puro+ and puro- plates:
	# (This will condense multiple replicates such that there will be a single x (puro-) and y (puro+) coordinate
	# for each well of each virus plate in each batch. The x/y coordinates are used to plot the infection efficiency
	# dot plot.)
	vPlateIds = vPlateGroups.keys()
	for vPlateId in vPlateIds:
		#print vPlateId
		plusPlateList = vPlateGroups[vPlateId]['puro+']
		meanPlus = averagePlates(plateSet, plusPlateList)
		minusPlateList = vPlateGroups[vPlateId]['puro-']
		meanMinus = averagePlates(plateSet, minusPlateList)

		vPlateGroups[vPlateId]['mean+'] = meanPlus
		vPlateGroups[vPlateId]['N+'] = len(plusPlateList)
		vPlateGroups[vPlateId]['mean-'] = meanMinus
		vPlateGroups[vPlateId]['N-'] = len(minusPlateList)

def averagePlates(plateSet, plateIdList):
	# averages the plates in a set of plates, well-by-well
	plateSum = {}

	# sum plates:
	for plateId in plateIdList:
		rawData = plateSet.getPlate(plateId).getRaw()
		for row in rawData.keys():
			plateSum.setdefault(row, {})
			for col in rawData[row].keys():
				plateSum[row].setdefault(col, 0.0)
				plateSum[row][col] += float(rawData[row][col])

	# divide by nPlates:
	nPlates = len(plateIdList)
	for row in plateSum.keys():
		for col in plateSum[row].keys():
			plateSum[row][col] /= nPlates

	return plateSum
