from javawrap import Color, BasicStroke
from javawrap.Ellipse2D import EllipseDouble
import MainHandle
import MajorObjects
import java.awt.Dimension as Dimension
import math
# Use jfreechar objects for all plotting functions
import org.jfree.chart.ChartFactory as ChartFactory
import org.jfree.chart.ChartPanel as ChartPanel
import org.jfree.chart.JFreeChart as JFreeChart
import org.jfree.chart.axis.CategoryLabelPosition as CategoryLabelPosition
import org.jfree.chart.axis.CategoryLabelPositions as CategoryLabelPositions
import org.jfree.chart.axis.CategoryLabelWidthType as CategoryLabelWidthType
import org.jfree.chart.axis.NumberAxis as NumberAxis
import org.jfree.chart.labels.StandardCategoryItemLabelGenerator as StandardCategoryItemLabelGenerator
import org.jfree.chart.plot.CombinedRangeCategoryPlot as CombinedRangeCategoryPlot
import org.jfree.chart.plot.PlotOrientation as PlotOrientation
import org.jfree.chart.renderer.category.BoxAndWhiskerRenderer as BoxAndWhiskerRenderer
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer as XYLineAndShapeRenderer
import org.jfree.data.category.DefaultCategoryDataset as DefaultCategoryDataset
import org.jfree.data.statistics.DefaultBoxAndWhiskerCategoryDataset
import org.jfree.data.xy.XYSeries as XYSeries
import org.jfree.data.xy.XYSeriesCollection as XYSeriesCollection
import org.jfree.text.TextBlockAnchor as TextBlockAnchor
import org.jfree.ui.RectangleAnchor as RectangleAnchor
import org.jfree.ui.TextAnchor as TextAnchor

masterObject = MainHandle.masterObject

# SCATTERPLOT
class NewScatterChart(ChartPanel):
	# extended jFreeChart ChartPanel with added features for making dot plots
	def __init__(self, title, xlabel, ylabel, width, height, seriesVisible=0):
		self.dataset = XYSeriesCollection()
		chart = ChartFactory.createScatterPlot(title, xlabel, ylabel, self.dataset,
 											   PlotOrientation.VERTICAL, 1, 0, 0)
		# create the scatter/dot plot in the panel
		ChartPanel.__init__(self, chart) #@UndefinedVariable
		self.setPreferredSize(Dimension(width, height))
		self.setSize(Dimension(width, height))
		self.setBackground(Color.WHITE)
		self.chart = chart

		# class variables:
		self.xyplot = chart.getXYPlot()       # data is plotted as an XYPlot
		self.renderer = XYLineAndShapeRenderer()
		# set renderer parameters
		self.renderer.setBaseLinesVisible(0)      # dots with no lines
		self.renderer.setBaseSeriesVisibleInLegend(seriesVisible) 
		self.renderer.setDrawSeriesLineAsPath(1)  # pointset is drawn as a single "path" without lines

		self.xyplot.setRenderer(self.renderer)
		# initialize lists and maps
		self.rendererList = []
		self.seriesIndexMap = {}
		self.seriesIdx = 0

		# initial values of slopes and cutoff
		self.upperLimitSlope = 4.0
		self.lowerLimitSlope = 0.25
		self.lowerLimitCutoff = 0.0
		masterObject.setField('IE lowCutoff', self.lowerLimitCutoff)

		# get the graphics object
		self.G = self.getGraphics()

	# chart title
	def setChartTitle(self, string):
		self.chart.setTitle(string)

	# x-axis label
	def setXAxisLabel(self, string):
		self.chart.getXYPlot().getDomainAxis().setLabel(string)

	# y-axis label
	def setYAxisLabel(self, string):
		self.chart.getXYPlot().getRangeAxis().setLabel(string)

	# dump pointset info (debugging)
	def dumpPointsetInfo(self, name):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return
		series = self.dataset.getSeries(sIdx)
		print 'Series dump: %s (idx=%d)' % (name, sIdx)
		print '  itemCount: %d' % series.getItemCount()
		print '  maxItemCount: %d' % series.getMaximumItemCount()
		print '  allow dup x: %s' % series.getAllowDuplicateXValues()
		print '  series visible: %s' % self.renderer.getSeriesVisible(sIdx)
		print '  lines visible: %s' % self.renderer.getSeriesLinesVisible(sIdx)
		print '  shapes visible: %s' % self.renderer.getSeriesShapesVisible(sIdx)
		print '  series shape: %s' % self.renderer.getSeriesShape(sIdx)
		print '  series paint: %s' % self.renderer.getSeriesPaint(sIdx)

	# add a pointset to the chart
	def addPointset(self, name, pointset, color=Color.BLUE):
		#print 'Adding pointset %s (idx: %d), #points: %d' % (name, self.seriesIdx,
		#													 pointset.getItemCount())
		self.dataset.addSeries(pointset)
		self.seriesIndexMap[name] = self.seriesIdx
		#print 'Adding series: %s, index: %d' % (name, self.seriesIdx)
		# set default dot size
		self.renderer.setSeriesShapesVisible(self.seriesIdx, 1)
		self.setPointsetShape(name, EllipseDouble(-2, -2, 3, 3))
		self.setPointsetColor(name, color)
		self.setPointsetAlpha(name, 1.0)
		self.setSeriesLineVisibility(name, 0)
		self.seriesIdx += 1

	# get a pointset by name
	def getPointsetSeries(self, name):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		return self.dataset.getSeries(sIdx)

	# set the line visibility of of pointset 
	def setSeriesLineVisibility(self, name, visible):
		try:
			# get one of the pointsets by name
			sIdx = self.seriesIndexMap[name]
		except:
			# if this isn't one of the virus plate pointsets, check to see if it
			# is one of the upper or lower limit triangles
			if name != 'upperLimit' and name != 'lowerLimit':
				# if not, flag an error and return
				print 'Unknown series name: %s' % name
			return

		# if it a valid pointset, set the visibility
		self.renderer.setSeriesLinesVisible(sIdx, visible)

	# set the line type using a java.awt BasicStroke
	def setSeriesLineStrokeDashed(self, name, dashArray=[3.0]):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		# BasicStroke(width, cap, join, miter limit, dash, dash phase)
		stroke = BasicStroke(1.0, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.0, dashArray, 0.0)

		# set the renderer stroke
		self.renderer.setSeriesStroke(sIdx, stroke)

	# set the legend visibility for a pointset
	def setSeriesLegendVisibility(self, name, visible):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		self.renderer.setSeriesVisibleInLegend(sIdx, visible)

	# set the shape visibility for a pointset
	def setSeriesShapeVisibility(self, name, visible):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		self.renderer.setSeriesShapesVisible(sIdx, visible)

	# get the map/dictionary of all pointsets
	def getPointsetMap(self):
		return self.seriesIndexMap

	# set the color of a pointset
	def setPointsetColor(self, name, color):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		self.renderer.setSeriesPaint(sIdx, color)
		#print 'Setting %s (idx: %d) to color %s' % (name, sIdx, color.toString())

	# sets the diameter of the dots for this pointset, and make the
	# dots centered on the data point location:
	def setPointsetSize(self, name, dia):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		try:
			d2 = float(dia) / 2.0
			self.renderer.setSeriesShape(sIdx, EllipseDouble(-d2, -d2, dia, dia))
		except:
			print 'Failed setting Pointset size for series %s (idx: %d)' % (name, sIdx)

	# set the shape of the points in a pointset
	def setPointsetShape(self, name, shape):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		try:
			self.renderer.setSeriesShape(sIdx, shape)
		except:
			print 'Failed setting Pointset shape for series %s (idx: %d)' % (name, sIdx)
		#print 'Setting %s (idx: %d) to shape %s' % (name, sIdx, shape.toString())

	# set the alpha (saturation) of the points in a pointset (1.0=full saturation, 0.0=invisible)
	def setPointsetAlpha(self, name, alpha):
		try:
			sIdx = self.seriesIndexMap[name]
		except:
			if name != 'upperLimit' and name != 'lowerLimit':
				print 'Unknown series name: %s' % name
			return

		try:
			# floating point versions of each color component
			red = self.renderer.getSeriesPaint(sIdx).getRed() / 255.0
			green = self.renderer.getSeriesPaint(sIdx).getGreen() / 255.0
			blue = self.renderer.getSeriesPaint(sIdx).getBlue() / 255.0
			# set the color with alpha 
			self.renderer.setSeriesPaint(sIdx, Color.newColor4(red, green, blue, alpha))
		except:
			print 'Failed setting Pointset alpha for series %s (idx: %d)' % (name, sIdx)
		#print 'Setting %s (idx: %d) to alpha %4.2f' % (name, sIdx, alpha)

	# draw the lower limit threshold triangle area
	def addLowerLimitTriangle(self):
		#self.squareAxes()
		# get the range and max of the x-axis
		xRange = self.xyplot.getDomainAxis().getRange()
		xMax = xRange.getUpperBound()

		if self.getPointsetSeries('lowerLimit') == None:
			# if this hasn't been drawn before, create it
			series = XYSeries('lowerLimit', 0, 1)
			self.addPointset('lowerLimit', series, Color.YELLOW)
		else:
			# otherwise, get the pointset
			series = self.getPointsetSeries('lowerLimit')
			series.clear()   # ... and clear it

		# create the pointset for this triangle:
		series = makeLowerLimitSeries(series, xMax, self.lowerLimitSlope, self.lowerLimitCutoff)

		# set its features
		self.setSeriesLineVisibility('lowerLimit', 1)
		self.setSeriesShapeVisibility('lowerLimit', 0)
		self.setPointsetAlpha('lowerLimit', 0.5)

		# and repaint it
		self.repaint()

	# draw the upper limit threshold triangle area
	def addUpperLimitTriangle(self):
		#self.squareAxes()   #  sets X and Y axes to cover the same range
		# get the range and max of the y-axis
		yRange = self.xyplot.getRangeAxis().getRange()
		yMax = yRange.getUpperBound()

		if self.getPointsetSeries('upperLimit') == None:
			# if this hasn't been drawn before, create it
			series = XYSeries('upperLimit', 0, 1)
			self.addPointset('upperLimit', series, Color.YELLOW)
		else:
			# otherwise, get the pointset
			series = self.getPointsetSeries('upperLimit')
			series.clear()   # and clear it

		# fill the series with the tiangle points
		series = makeUpperLimitSeries(series, yMax, self.upperLimitSlope)

		# set its features
		self.setSeriesLineVisibility('upperLimit', 1)
		self.setSeriesShapeVisibility('upperLimit', 0)
		self.setPointsetAlpha('upperLimit', 0.5)

		# and repaint it
		self.repaint()

	# hide the lower limit triangle by setting its alpha to 0.0
	def hideLowerLimitTriangle(self):
		self.setPointsetAlpha('lowerLimit', 0.0)

	# show the lower limit triangle by setting its alpha to 0.5
	def showLowerLimitTriangle(self):
		self.setPointsetAlpha('lowerLimit', 0.5)
		self.repaint()

	# hide the upper limit triangle by setting its alpha to 0.0
	def hideUpperLimitTriangle(self):
		self.setPointsetAlpha('upperLimit', 0.0)

	# hide the upper limit triangle by setting its alpha to 0.5
	def showUpperLimitTriangle(self):
		self.setPointsetAlpha('upperLimit', 0.5)
		self.repaint()

	# set the slope of the lower limit triangle
	def setLowerLimitSlope(self, slope):
		series = self.getPointsetSeries('lowerLimit')
		if series == None:
			return
		series.clear()

		self.lowerLimitSlope = slope
		self.addLowerLimitTriangle()
		# indicate that the selected plateset analysis is no longer valid:
		masterObject.setField('Update analysis', 1)
		# indicate that hairpin scores need to be updated:
		masterObject.setField('Compute hairpin scores', 1)

	# set the absolute cutoff of the lower limit triangle
	def setLowerLimitCutoff(self, cutoff):
		#print 'setLowerLimitCutoff: ', cutoff
		series = self.getPointsetSeries('lowerLimit')
		if series == None:
			return
		series.clear()

		self.lowerLimitCutoff = cutoff
		masterObject.setField('IE lowCutoff', cutoff)
		self.addLowerLimitTriangle()
		# indicate that the selected plateset analysis is no longer valid:
		masterObject.setField('Update analysis', 1)
		# indicate that hairpin scores need to be updated:
		masterObject.setField('Compute hairpin scores', 1)

	# set the slope of the upper limit triangle
	def setUpperLimitSlope(self, slope):
		series = self.getPointsetSeries('upperLimit')
		if series == None:
			return
		series.clear()

		self.upperLimitSlope = slope
		self.addUpperLimitTriangle()
		# indicate that the selected plateset analysis is no longer valid:
		masterObject.setField('Update analysis', 1)
		# indicate that hairpin scores need to be updated:
		masterObject.setField('Compute hairpin scores', 1)

	# draw the 100% infection efficiency guide line as a light gray line
	# with a 45-degree slope
	def addUnitSlopeLine(self):
		plotRect = self.getScreenDataArea()
		#print 'Plot rectangle:', plotRect
		xAx = self.xyplot.getDomainAxis()
		yAx = self.xyplot.getRangeAxis()
		# add a 1% buffer to the unitSlopeLine:
		axMax = self.dataset.getDomainUpperBound(0) * 1.01
		#axMax = max(xAx.getRange().getUpperBound(),yAx.getRange().getUpperBound())
		seriesName = 'unitSlopeLine_Limit'
		series = XYSeries(seriesName)
		series.add(0.0, 0.0)
		series.add(axMax, axMax)
		self.addPointset(seriesName, series, Color.BLACK)
		self.setSeriesLineVisibility(seriesName, 1)
		self.setSeriesShapeVisibility(seriesName, 0)
		self.setPointsetAlpha(seriesName, 0.25)

	# set the axes to be square (i.e., x- and y-axes have the same range
	def squareAxes(self, axMax=None):
		xAx = self.xyplot.getDomainAxis()
		yAx = self.xyplot.getRangeAxis()
		if axMax == None:
			axMax = self.dataset.getDomainUpperBound(0)
			#axMax = max(xAx.getRange().getUpperBound(),yAx.getRange().getUpperBound())

		xAx.setRange(0.0, axMax)
		yAx.setRange(0.0, axMax)
		self.setUpperLimitSlope(self.upperLimitSlope)
		self.setLowerLimitSlope(self.lowerLimitSlope)

	# clear all data from the chart
	def clearChart(self):
		seriesList = []
		for series in self.seriesIndexMap.keys():
			seriesIdx = self.seriesIndexMap[series]
			seriesList.append(seriesIdx)
			del self.seriesIndexMap[series]

		# sort the list of indices, then reverse it so that the highest indexed
		# items are removed first:
		#seriesList.sort()
		#seriesList.reverse()
		#for seriesIdx in seriesList:
		#	self.dataset.removeSeries(seriesIdx)

		self.dataset.removeAllSeries()	  # more straightforward
		self.seriesIdx = 0

# HISTOGRAM
class NewHistoChart(ChartPanel):
	# extended jFreeChart ChartPanel with added features for making histograms
	def __init__(self, title, xlabel, ylabel, width, height):
		#self.dataset = XYSeriesCollection()
		self.dataset = DefaultCategoryDataset()
		chart = ChartFactory.createBarChart(title, xlabel, ylabel, self.dataset,
 											   PlotOrientation.VERTICAL, 1, 0, 0)
		# create the histogram/bar chart in the panel
		ChartPanel.__init__(self, chart) #@UndefinedVariable
		self.setPreferredSize(Dimension(width, height))
		#self.setSize(Dimension(width, height))
		self.setBackground(Color.WHITE)
		self.chart = chart

		# class variables:
		self.xyplot = chart.getCategoryPlot()     # data is plotted as a Category plot
		self.renderer = self.xyplot.getRenderer()
		self.renderer.setItemMargin(0.0)
		self.renderer.setDrawBarOutline(0)
		yAxis = self.chart.getPlot().getRangeAxis()
		yAxis.setNumberFormatOverride(MajorObjects.new2point2Format())

		#self.xyplot.setRenderer(self.renderer)
		self.rendererList = []
		self.seriesIndexMap = {}
		self.seriesIdx = 0

	# clear the chart
	def resetPointset(self):
		self.dataset.clear()

	# each row in the pointList has the form: <binLabel>,value
	def addPointset(self, name, pointList, color=Color.BLUE):
		for point in pointList:
			self.dataset.addValue(point[1], name, point[0])
		self.seriesIndexMap[name] = self.seriesIdx

		# set the bar color
		self.renderer.setSeriesPaint(self.seriesIdx, color)

		# increment the series index
		self.seriesIdx += 1

	# get all pointsets
	def getPointsetMap(self):
		return self.seriesIndexMap

	# set the color of a pointset by name
	def setPointsetColor(self, name, color):
		try:
			sIdx = self.seriesIndexMap[name]
			self.renderer.setSeriesPaint(sIdx, color)
		except:
			print 'Failed setting %s (idx: %d) to color %s' % (name, sIdx, color.toString())
			#pass


# BOX_WHISKER (box-and-whisker plot)
#
# Usage:
#  bwPlot = BoxAndWhiskerChartCombo(...)

class BoxAndWhiskerChartCombo(ChartPanel):
	# extended jFreeChart ChartPanel with added features for making box-and-whisker plots
	# with multiple charts displayed side-by-side
	def __init__(self, title, ylabel, width, height, legend=1, angle=60):
		self.comboPlot = CombinedRangeCategoryPlot()  # use CombinedRangeCategoryPlot as the plot type
		self.rangeAxis = NumberAxis(ylabel)
		self.ylabel = ylabel
		self.comboPlot.setRangeAxis(self.rangeAxis)
		self.angle = angle

		# initialize lists/maps
		self.bwPlot = {}
		self.plotIdx = []
		self.plotList = []

		# create the plot
		chart = JFreeChart(self.comboPlot)
		if legend == 0:
			chart.removeLegend()

		ChartPanel.__init__(self, chart) #@UndefinedVariable
		self.chart = chart

		self.setPreferredSize(Dimension(width, height))
		self.setSize(self.getPreferredSize())
		self.renderer = None

    # add a box and whisket plot to the panel
	def addBoxAndWhiskerPlot(self, xlabel, dataMap, colorMap, seriesList, categories, catFill=None, weight=1):
		for plate in dataMap.keys():
			for cat in dataMap[plate].keys():
				if len(dataMap[plate][cat]) == 0:
					dataMap[plate][cat] = [0]

		# subplots are indexed by 'xlabel'
		self.plotIdx.append(xlabel)
		self.bwPlot[xlabel] = BWObject(xlabel)
		self.bwPlot[xlabel].addDataMap(dataMap, seriesList, categories)
		self.bwPlot[xlabel].setLabelRotation(self.angle)
		self.comboPlot.add(self.bwPlot[xlabel].plot, weight)
		self.plotList.append(self.bwPlot[xlabel].plot)
		renderer = self.bwPlot[xlabel].renderer
		for series in seriesList:
			for category in categories:
				renderer.setPaintFor(categories.index(category), seriesList.index(series),
									 colorMap[series][category])

		#renderer.dumpColors()

	# set the angle to display the axis labels
	def setLabelRotation(self, angle):
		self.angle = angle
		for xlabel in self.plotIdx:
			self.bwPlot[xlabel].setLabelRotation(angle)

	# resize the label font by a fraction
	def resizeLabelFont(self, fraction):
		self.fraction = fraction
		for xlabel in self.plotIdx:
			self.bwPlot[xlabel].resizeLabelFont(fraction)

	# reset/clear the chart
	def resetChart(self):
		# return if there are no subplots:
		if len(self.comboPlot.getSubplots()) == 0:
			return

		# remove any subplots and reset the storage parameters:
		plotList = []
		for subPlot in self.comboPlot.getSubplots():
			plotList.append(subPlot)

		for subPlot in plotList:
			self.comboPlot.remove(subPlot)

		self.bwPlot = {}
		self.plotIdx = []
		self.plotList = []

	# AD: not sure what ths did, but it's not used
	def adjustWeights(self):
		total = 0
		for xlabel in self.plotIdx:
			total += self.bwPlot[xlabel].seriesCount()

		for xlabel in self.plotIdx:
			nCat = self.bwPlot[xlabel].seriesCount()
			wgt = int(total / nCat)
			self.bwPlot[xlabel].plot.setWeight(wgt)
			#print 'Weights: ',xlabel, wgt

# the box-and-whiskers plot object
class BWObject:
	def __init__(self, xlabel):
		# use the DefaultBoxAndWhiskerCategoryDataset as the dataset
		self.dataset = org.jfree.data.statistics.DefaultBoxAndWhiskerCategoryDataset()
		# make an empty box-and-whiskers plot
		self.plot = makeBareBoxAndWhiskerPlot(xlabel, self.dataset)
		self.renderer = self.plot.getRenderer()
		self.dataMap = None
		self.seriesList = None
		self.categories = None

	# add a dataset to the plot
	def datasetAdd(self, item, series, category):
		self.dataset.add(item, category, series)

	# add datasets in a particular order
	def addDataMap(self, dataMap, seriesList, categories):
		# series/category lists just to allow specific order of series/category:
		self.dataMap = dataMap
		self.seriesList = seriesList
		self.categories = categories
		for series in seriesList:
			for category in categories:
				self.dataset.add(dataMap[series][category], category, series)

	# count the number of datasets that are being displayed
	def seriesCount(self):
		return len(self.dataMap.keys())

	# resize the axis labels by a fraction
	def resizeLabelFont(self, fraction):
		axis = self.plot.domainAxis
		font = axis.getTickLabelFont()
		fontSize = font.getSize2D()
		newSize = fontSize * fraction
		newFont = font.deriveFont(newSize)
		axis.setTickLabelFont(newFont)

	# set the angle (rotation) of the label text
	def setLabelRotation(self, angle):
		axis = self.plot.domainAxis
		positions = axis.getCategoryLabelPositions()
		position = CategoryLabelPosition(RectangleAnchor.TOP,
										 TextBlockAnchor.CENTER_LEFT,
										 TextAnchor.TOP_CENTER,
										 angle * math.pi / 180.0,
										 CategoryLabelWidthType.CATEGORY,
										 50)
		axis.setCategoryLabelPositions(CategoryLabelPositions.replaceBottomPosition(positions, position))

# make an empty box-and-whisker plot
def makeBareBoxAndWhiskerPlot(xlabel, dataset):
	disposableChart = ChartFactory.createBoxAndWhiskerChart('', xlabel, '', dataset, 0)
	disposableChart.getPlot().setRenderer(customBoxAndWhiskerRenderer())
	return disposableChart.getPlot()

class NewBoxAndWhiskerChart(ChartPanel):
	# one of a possible set of box-and-whisker plots plotted side-by-side 
	def __init__(self, title, xlabel, ylabel, width, height, dataMap=None, seriesList=None, categories=None):
		self.dataset = org.jfree.data.statistics.DefaultBoxAndWhiskerCategoryDataset()
		self.dataMap = dataMap
		self.seriesList = seriesList
		self.categories = categories

		# add all of the datasets in the series list
		if self.dataMap:
			for series in seriesList:
				for category in categories:
					self.dataset.add(self.dataMap[series][category], category, series)

		# create the chart object
		chart = ChartFactory.createBoxAndWhiskerChart(title, xlabel, ylabel, self.dataset, 1)
		ChartPanel.__init__(self, chart) #@UndefinedVariable
		self.setPreferredSize(Dimension(width, height))
		self.setSize(self.getPreferredSize())
		#self.getChart().getCategoryPlot().setRenderer(BoxAndWhiskerRendererWithLimiting())
		self.renderer = self.getChart().getCategoryPlot().getRenderer()
		#self.chart.plot.domainAxis.setCategoryLabelPositionOffset(5)

	# set the chart data
	def setData(self, dataMap, seriesList, categories):
		self.dataMap = dataMap
		self.dataset = org.jfree.data.statistics.DefaultBoxAndWhiskerCategoryDataset()
		self.getChart().getPlot().setDataset(self.dataset)

		# add all of the datasets in the series list
		for series in seriesList:
			for category in categories:
				self.dataset.add(dataMap[series][category], category, series)

	# set the color for the type
	def setTypeColor(self, typeIndex, color):
		self.renderer.setSeriesPaint(typeIndex, color, 1)

	# set the label text rotation (angle)
	def setLabelRotation(self, angle):
		axis = self.chart.plot.domainAxis
		positions = axis.getCategoryLabelPositions()
		position = CategoryLabelPosition(RectangleAnchor.TOP,
										 TextBlockAnchor.CENTER_LEFT,
										 TextAnchor.TOP_CENTER,
										 angle * math.pi / 180.0,
										 CategoryLabelWidthType.CATEGORY,
										 50)
		axis.setCategoryLabelPositions(CategoryLabelPositions.replaceBottomPosition(positions, position))

# AD: apparently not used
class BoxAndWhiskerRendererWithLimiting(BoxAndWhiskerRenderer):
	def __init__(self):
		BoxAndWhiskerRenderer.__init__(self) #@UndefinedVariable
		self.getGraphics()
		#print 'BoxAndWhiskerRenderer: ', self.initialise()

# AD: apparently not used
class CategoryItemTextLabelGenerator(StandardCategoryItemLabelGenerator):
	def __init__(self, lText):
		StandardCategoryItemLabelGenerator.__init__(self) #@UndefinedVariable
		self.lText = lText

	def generateLabel(self, dataset, series, category):
		#return self.lText
		print 'generate label: ', series, category
		return 'X'

class customBoxAndWhiskerRenderer(BoxAndWhiskerRenderer):
	# extended BoxAndWhiskerRenderer with storage for various color-related features
	def __init__(self, defaultColor=None):
		if defaultColor is None:
			defaultColor = Color.GRAY

		self.defaultColor = defaultColor

		self.colors = {}
		self.defaultRowColors = {}
		self.defaultColColors = {}

	# set the paint color for a series/category
	def setPaintFor(self, series, category, color):
		self.colors[series] = self.colors.get(series, {})
		self.colors[series][category] = color
		#print 'Setting color:', series, category, color

	# Dump the present color assignments (debugging)
	def dumpColors(self):
		print 'Dumping colors:::::'
		for series in self.colors.keys():
			for category in self.colors[series].keys():
				print series, category, self.colors[series][category]
		print 'Done dumping colors'

	# set the color for a series
	def setSeriesPaint(self, series, color):
		for key in self.colors.get(series, {}).keys():
			self.colors[series][key] = color
		self.defaultRowColors[series] = color

	# set the color for a category
	def setCategoryPaint(self, category, color):
		for key in self.colors.keys():
			self.colors[key][category] = color
			self.defaultColColors[category] = color

	# get the paint settings for a data item
	def getItemPaint(self, row, col):
		"""Returns self.colors[row][col];
		falls back to self.defaultColColors[col];
		falls back to self.defaultRowColors[row];
		if necessary, falls back to self.defaultColor."""
		#print 'row:col:color',row,col,self.colors.get(row,{}).get(col,self.defaultColColors.get(col,self.defaultRowColors.get(row,self.defaultColor)))
		return self.colors.get(row, {}).get(col, self.defaultColColors.get(col, self.defaultRowColors.get(row, self.defaultColor)))

###################################################################################################
# NOTE: makeLowerLimitSeries() and makeUpperLimitSeries() create datasets that, when drawn,
# have the appearance of a filled in triangle. The first version of this tried to use a solid 
# triangle Shape object, but re-drawing the triangle when the threshold slider was moved
# was too sluggish. Drawing 100 lines wsa much quicker than erasing and repainting a triangle.
###################################################################################################
def makeLowerLimitSeries(series, xMax, slope, cutoff):
	nLines = 100
	yMax = xMax * slope
	xMin = cutoff
	yMin = xMin * slope
	yStepSize = yMax / nLines
	xStepSize = xMax / nLines
	series.add(xMin, yMin)
	series.add(xMax, yMax)
	for i in range(nLines):
		x = xMax - (xStepSize * i)
		if x < xMin:
			continue
		y = yMax - (yStepSize * i)
		series.add(x, y)
		series.add(x, 0)

	return series

def makeUpperLimitSeries(series, yMax, slope):
	nLines = 100
	xMax = yMax / slope
	yStepSize = yMax / nLines
	xStepSize = xMax / nLines
	series.add(0.0, 0.0)
	series.add(xMax, yMax)
	for i in range(nLines):
		x = xMax - (xStepSize * i)
		y = yMax - (yStepSize * i)
		series.add(x, y)
		series.add(0, y)

	return series
